import axios from 'axios';

const API_URL = process.env.NEXT_PUBLIC_API_URL || 'https://admin.captaincity.net/api';
const api = axios.create({
  baseURL: API_URL,
  headers: {
    'Accept': 'application/json',
  },
});

// Add interceptor to include auth token
api.interceptors.request.use((config) => {
  const token = localStorage.getItem('token'); // Assuming token is stored in localStorage
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

export const AuthService = {
    login: (credentials: any) => api.post('/login', credentials),
    logout: () => api.post('/logout'),
    user: () => api.get('/user'),
};

export const UserService = {
  getAll: (params?: any) => api.get('/admin/users', { params }),
  getOne: (id: string | number) => api.get(`/admin/users/${id}`),
  create: (data: any) => api.post('/admin/users', data),
  update: (id: string | number, data: any) => {
    if (data instanceof FormData) {
      if (!data.has('_method')) {
        data.append('_method', 'PUT');
      }
      return api.post(`/admin/users/${id}`, data);
    }
    return api.put(`/admin/users/${id}`, data);
  },
  delete: (id: string | number) => api.delete(`/admin/users/${id}`),
};

export const StoreService = {
  getAll: (params?: any) => api.get('/admin/vendors', { params }),
  getOne: (id: string | number) => api.get(`/admin/vendors/${id}`),
  create: (data: any) => api.post('/admin/vendors', data),
  update: (id: string | number, data: any) => api.put(`/admin/vendors/${id}`, data),
  delete: (id: string | number) => api.delete(`/admin/vendors/${id}`),
};

export const ProductService = {
  getAll: (params?: any) => api.get("/admin/products", { params }),
  getOne: (id: string) => api.get(`/admin/products/${id}`),
  create: (data: any) => api.post("/admin/products", data),
  update: (id: string, data: any) => api.put(`/admin/products/${id}`, data),
  delete: (id: string) => api.delete(`/admin/products/${id}`),
};

// Category Service
export const CategoryService = {
  getAll: (params?: any) => api.get("/admin/categories", { params }),
  getOne: (id: string) => api.get(`/admin/categories/${id}`),
  create: (data: any) => api.post("/admin/categories", data),
  update: (id: string, data: any) => api.put(`/admin/categories/${id}`, data),
  delete: (id: string) => api.delete(`/admin/categories/${id}`),
};

// Order Service
export const OrderService = {
  getAll: (params?: any) => api.get("/admin/orders", { params }),
  getOne: (id: string) => api.get(`/admin/orders/${id}`),
  updateStatus: (id: string, status: string) => api.put(`/admin/orders/${id}/status`, { status }),
  delete: (id: string) => api.delete(`/admin/orders/${id}`),
  getActiveLocations: () => api.get("/admin/dashboard/active-order-locations"),
};

// Ride Service
export const RideService = {
  getAll: (params?: any) => api.get("/admin/ride/rides", { params }),
  getOne: (id: string) => api.get(`/admin/ride/rides/${id}`),
  updateStatus: (id: string, status: string) => api.put(`/admin/ride/rides/${id}`, { status }),
  delete: (id: string) => api.delete(`/admin/ride/rides/${id}`),
};

export const DriverService = {
  getAll: (params?: any) => api.get("/admin/ride/drivers", { params }),
  getOne: (id: string) => api.get(`/admin/ride/drivers/${id}`),
  create: (data: any) => api.post("/admin/ride/drivers", data),
  update: (id: string, data: any) => api.put(`/admin/ride/drivers/${id}`, data),
  delete: (id: string) => api.delete(`/admin/ride/drivers/${id}`),
};

export const VehicleTypeService = {
  getAll: (params?: any) => api.get("/admin/ride/vehicletypes", { params }),
  getOne: (id: string | number) => api.get(`/admin/ride/vehicletypes/${id}`),
  create: (data: FormData) => api.post("/admin/ride/vehicletypes", data),
  update: (id: string | number, data: FormData) => {
    if (!data.has("_method")) {
      data.append("_method", "PUT");
    }
    return api.post(`/admin/ride/vehicletypes/${id}`, data);
  },
  delete: (id: string | number) => api.delete(`/admin/ride/vehicletypes/${id}`),
};

export const BannerService = {
  getAll: (params?: any) => api.get("/admin/banners", { params }),
  getOne: (id: string | number) => api.get(`/admin/banners/${id}`),
  create: (data: FormData) => api.post("/admin/banners", data),
  update: (id: string | number, data: FormData) => {
    if (!data.has("_method")) {
      data.append("_method", "PUT");
    }
    return api.post(`/admin/banners/${id}`, data);
  },
  delete: (id: string | number) => api.delete(`/admin/banners/${id}`),
};

export const CouponService = {
  getAll: (params?: any) => api.get("/admin/coupons", { params }),
  getOne: (id: string | number) => api.get(`/admin/coupons/${id}`),
  create: (data: any) => api.post("/admin/coupons", data),
  update: (id: string | number, data: any) => api.put(`/admin/coupons/${id}`, data),
  delete: (id: string | number) => api.delete(`/admin/coupons/${id}`),
};

export const LocationChargeService = {
  getAll: (params?: any) => api.get("/admin/ride/location-charges", { params }),
  getOne: (id: string | number) => api.get(`/admin/ride/location-charges/${id}`),
  create: (data: any) => api.post("/admin/ride/location-charges", data),
  update: (id: string | number, data: any) => api.put(`/admin/ride/location-charges/${id}`, data),
  delete: (id: string | number) => api.delete(`/admin/ride/location-charges/${id}`),
};

export const ProviderService = {
  getAll: (params?: any) => api.get("/admin/providers", { params }),
  getOne: (id: string | number) => api.get(`/admin/providers/${id}`),
  create: (data: any) => api.post("/admin/providers", data),
  update: (id: string | number, data: any) => api.put(`/admin/providers/${id}`, data),
  delete: (id: string | number) => api.delete(`/admin/providers/${id}`),
};

export const AppointmentService = {
  getAll: (params?: any) => api.get("/admin/provider/appointments", { params }),
  getOne: (id: string | number) => api.get(`/admin/provider/appointments/${id}`),
  update: (id: string | number, data: any) => api.put(`/admin/provider/appointments/${id}`, data),
  updateStatus: (id: string | number, status: string) => api.put(`/admin/provider/appointments/${id}`, { status }),
  delete: (id: string | number) => api.delete(`/admin/provider/appointments/${id}`),
};

export const TransactionService = {
  getAll: (params?: any) => api.get("/admin/wallet/transactions", { params }),
  getOne: (id: string | number) => api.get(`/admin/wallet/transactions/${id}`),
  getAnalytics: (params?: any) => api.get("/admin/dashboard/transaction-analytics", { params }),
};

export const PayoutService = {
  getAll: (params?: any) => api.get("/admin/wallet/transactions", { params: { ...params, type: 'payout' } }),
  updateStatus: (id: string | number, confirmed: boolean) => {
      const data = new FormData();
      data.append('is_paid', confirmed ? '1' : '0');
      data.append('_method', 'PUT');
      return api.post(`/admin/wallet/transactions/${id}`, data);
  },
};

export default api;
