"use client";

import React, { useState, useCallback, memo } from 'react';
import { GoogleMap, useJsApiLoader, Marker, InfoWindow } from '@react-google-maps/api';
import { Loader2, Store, User, Package, MapPin } from 'lucide-react';

const containerStyle = {
  width: '100%',
  height: '500px',
  borderRadius: '16px',
};

// Default center (Cairo as a fallback)
const defaultCenter = {
  lat: 30.0444,
  lng: 31.2357
};

interface OrderLocation {
  id: number;
  status: string;
  total: number;
  vendor?: {
    name: string;
    latitude: number;
    longitude: number;
  };
  address?: {
    latitude: number;
    longitude: number;
    formatted_address: string;
  };
  source_address?: {
    latitude: number;
    longitude: number;
    formatted_address: string;
  };
}

interface OrderMapProps {
  orders: OrderLocation[];
}

function OrderMap({ orders }: OrderMapProps) {
  const { isLoaded } = useJsApiLoader({
    id: 'google-map-script',
    googleMapsApiKey: process.env.NEXT_PUBLIC_GOOGLE_MAPS_API_KEY || ""
  });

  const [map, setMap] = useState<google.maps.Map | null>(null);
  const [selectedOrder, setSelectedOrder] = useState<OrderLocation | null>(null);
  const [infoWindowType, setInfoWindowType] = useState<'store' | 'customer' | null>(null);

  const onLoad = useCallback(function callback(map: google.maps.Map) {
    if (orders.length > 0) {
      const bounds = new window.google.maps.LatLngBounds();
      orders.forEach(order => {
        if (order.source_address?.latitude && order.source_address?.longitude) {
          bounds.extend({ lat: Number(order.source_address.latitude), lng: Number(order.source_address.longitude) });
        }
        if (order.address?.latitude && order.address?.longitude) {
          bounds.extend({ lat: Number(order.address.latitude), lng: Number(order.address.longitude) });
        }
      });
      map.fitBounds(bounds);
    }
    setMap(map);
  }, [orders]);

  const onUnmount = useCallback(function callback(map: google.maps.Map) {
    setMap(null);
  }, []);

  if (!isLoaded) {
    return (
      <div className="card-custom d-flex flex-column align-items-center justify-content-center" style={{ height: '500px' }}>
        <Loader2 className="text-primary spin-animation mb-2" size={32} />
        <p className="text-muted-custom small">Loading Maps...</p>
      </div>
    );
  }

  return (
    <div className="position-relative">
      <GoogleMap
        mapContainerStyle={containerStyle}
        center={defaultCenter}
        zoom={12}
        onLoad={onLoad}
        onUnmount={onUnmount}
        options={{
          styles: [
            {
              "featureType": "all",
              "elementType": "geometry.fill",
              "stylers": [{ "weight": "2.00" }]
            },
            {
              "featureType": "all",
              "elementType": "geometry.stroke",
              "stylers": [{ "color": "#9c9c9c" }]
            },
            {
              "featureType": "all",
              "elementType": "labels.text",
              "stylers": [{ "visibility": "on" }]
            },
            {
              "featureType": "landscape",
              "elementType": "all",
              "stylers": [{ "color": "#f2f2f2" }]
            },
            {
              "featureType": "landscape",
              "elementType": "geometry.fill",
              "stylers": [{ "color": "#ffffff" }]
            },
            {
              "featureType": "landscape.man_made",
              "elementType": "geometry.fill",
              "stylers": [{ "color": "#ffffff" }]
            },
            {
              "featureType": "poi",
              "elementType": "all",
              "stylers": [{ "visibility": "off" }]
            },
            {
              "featureType": "road",
              "elementType": "all",
              "stylers": [{ "saturation": -100 }, { "lightness": 45 }]
            },
            {
              "featureType": "road",
              "elementType": "geometry.fill",
              "stylers": [{ "color": "#eeeeee" }]
            },
            {
              "featureType": "road",
              "elementType": "labels.text.fill",
              "stylers": [{ "color": "#7b7b7b" }]
            },
            {
              "featureType": "road",
              "elementType": "labels.text.stroke",
              "stylers": [{ "color": "#ffffff" }]
            },
            {
              "featureType": "road.highway",
              "elementType": "all",
              "stylers": [{ "visibility": "simplified" }]
            },
            {
              "featureType": "road.arterial",
              "elementType": "labels.icon",
              "stylers": [{ "visibility": "off" }]
            },
            {
              "featureType": "transit",
              "elementType": "all",
              "stylers": [{ "visibility": "off" }]
            },
            {
              "featureType": "water",
              "elementType": "all",
              "stylers": [{ "color": "#46bcec" }, { "visibility": "on" }]
            },
            {
              "featureType": "water",
              "elementType": "geometry.fill",
              "stylers": [{ "color": "#c8d7d4" }]
            },
            {
              "featureType": "water",
              "elementType": "labels.text.fill",
              "stylers": [{ "color": "#070707" }]
            },
            {
              "featureType": "water",
              "elementType": "labels.text.stroke",
              "stylers": [{ "color": "#ffffff" }]
            }
          ]
        }}
      >
        {orders.map((order) => (
          <React.Fragment key={order.id}>
            {/* Store Marker */}
            {order.source_address?.latitude && (
              <Marker
                position={{ lat: Number(order.source_address.latitude), lng: Number(order.source_address.longitude) }}
                onClick={() => {
                  setSelectedOrder(order);
                  setInfoWindowType('store');
                }}
                icon={{
                  url: 'https://maps.google.com/mapfiles/ms/icons/blue-dot.png'
                }}
              />
            )}
            {/* Customer Marker */}
            {order.address?.latitude && (
              <Marker
                position={{ lat: Number(order.address.latitude), lng: Number(order.address.longitude) }}
                onClick={() => {
                  setSelectedOrder(order);
                  setInfoWindowType('customer');
                }}
                icon={{
                  url: 'https://maps.google.com/mapfiles/ms/icons/red-dot.png'
                }}
              />
            )}
          </React.Fragment>
        ))}

        {selectedOrder && infoWindowType && (
          <InfoWindow
            position={
              infoWindowType === 'store'
                ? { lat: Number(selectedOrder.source_address!.latitude), lng: Number(selectedOrder.source_address!.longitude) }
                : { lat: Number(selectedOrder.address!.latitude), lng: Number(selectedOrder.address!.longitude) }
            }
            onCloseClick={() => {
              setSelectedOrder(null);
              setInfoWindowType(null);
            }}
          >
            <div className="p-1" style={{ maxWidth: '200px' }}>
              <div className="d-flex align-items-center gap-2 mb-2 pb-2 border-bottom">
                {infoWindowType === 'store' ? <Store size={16} className="text-primary" /> : <User size={16} className="text-danger" />}
                <span className="fw-bold small">Order #{selectedOrder.id}</span>
              </div>
              <div className="d-flex flex-column gap-1">
                <div className="d-flex align-items-center gap-1 small text-muted">
                  <Package size={12} />
                  <span>Status: {selectedOrder.status}</span>
                </div>
                <div className="d-flex align-items-center gap-1 small text-muted">
                  <MapPin size={12} />
                  <span className="text-truncate">
                    {infoWindowType === 'store' ? selectedOrder.source_address?.formatted_address : selectedOrder.address?.formatted_address}
                  </span>
                </div>
                <div className="fw-bold text-success small mt-1">
                  Total: ${Number(selectedOrder.total).toFixed(2)}
                </div>
              </div>
            </div>
          </InfoWindow>
        )}
      </GoogleMap>
      
      {!process.env.NEXT_PUBLIC_GOOGLE_MAPS_API_KEY && (
        <div className="position-absolute top-0 start-0 w-100 h-100 d-flex align-items-center justify-content-center" style={{ backgroundColor: 'rgba(255,255,255,0.8)', zIndex: 1 }}>
           <div className="text-center p-4 bg-white shadow rounded-3 border">
              <p className="text-danger fw-bold mb-1">Google Maps API Key Missing</p>
              <p className="small text-muted mb-0">Please add `NEXT_PUBLIC_GOOGLE_MAPS_API_KEY` to your environment variables.</p>
           </div>
        </div>
      )}
    </div>
  );
}

export default memo(OrderMap);
