"use client";

import PageHeader from "@/components/admin/PageHeader";
import { ArrowLeftRight, Search, CreditCard, User, Mail, DollarSign } from "lucide-react";
import { useState, useEffect } from "react";
import { TransactionService } from "@/services/api";
import Pagination from "@/components/admin/Pagination";
import { useDebounce } from "@/hooks/useDebounce";
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, AreaChart, Area } from "recharts";
import Link from "next/link";
import { Eye } from "lucide-react";

export default function TransactionsPage() {
  const [transactions, setTransactions] = useState([]);
  const [loading, setLoading] = useState(true);
  const [pagination, setPagination] = useState<any>(null);
  const [page, setPage] = useState(1);
  const [search, setSearch] = useState("");
  const debouncedSearch = useDebounce(search, 500);
  const [typeFilter, setTypeFilter] = useState("");
  const [analytics, setAnalytics] = useState<any>(null);

  useEffect(() => {
    fetchTransactions();
    fetchAnalytics();
  }, [page, debouncedSearch, typeFilter]);

  const fetchTransactions = async () => {
    setLoading(true);
    try {
      const params: any = { page };
      if (debouncedSearch) params.search = debouncedSearch;
      if (typeFilter) params.type = typeFilter;
      
      const response = await TransactionService.getAll(params);
      setTransactions(response.data.data);
      setPagination(response.data.meta);
    } catch (error) {
      console.error("Failed to fetch transactions:", error);
    } finally {
      setLoading(false);
    }
  };

  const fetchAnalytics = async () => {
      try {
          const response = await TransactionService.getAnalytics({ period: 'week' });
          setAnalytics(response.data);
      } catch (error) {
          console.error("Failed to fetch analytics", error);
      }
  }

  const getAmountColor = (type: string) => {
    return type === 'deposit' ? 'text-success' : 'text-danger';
  };

  const prepareChartData = () => {
    if (!analytics || !analytics.chart) return [];
    
    return analytics.chart.chartLabel.map((label: string, index: number) => ({
      name: label,
      amount: parseFloat(analytics.chart.linesData[0][index] || 0)
    }));
  };

  return (
    <div className="py-2">
      <PageHeader title="Transactions" description="View system transactions (deposits & withdrawals)." />

      {/* Analytics Section */}
      {analytics && (
      <div className="mb-4">
          <div className="row g-3 mb-4">
              {analytics.summary.map((item: any, index: number) => (
                  <div className="col-md-3 col-6" key={index}>
                      <div className="card-custom p-3 text-center border-0 shadow-sm">
                        <h6 className="text-secondary small text-uppercase mb-2">{item.title}</h6>
                        <div className="d-flex align-items-center justify-content-center gap-1">
                             <span className="text-muted small">$</span>
                             <h3 className="fw-bold mb-0 text-dark">{parseFloat(item.value).toLocaleString()}</h3>
                        </div>
                      </div>
                  </div>
              ))}
          </div>

          {/* Chart */}
          <div className="card-custom p-4 border-0 shadow-sm mb-4">
             <h5 className="fw-bold mb-4">Transactions Overview (Last 7 Days)</h5>
             <div style={{ width: '100%', height: 300 }}>
                <ResponsiveContainer>
                    <AreaChart data={prepareChartData()}>
                        <defs>
                            <linearGradient id="colorAmount" x1="0" y1="0" x2="0" y2="1">
                            <stop offset="5%" stopColor="#8884d8" stopOpacity={0.8}/>
                            <stop offset="95%" stopColor="#8884d8" stopOpacity={0}/>
                            </linearGradient>
                        </defs>
                        <CartesianGrid strokeDasharray="3 3" vertical={false} stroke="#E5E7EB" />
                        <XAxis dataKey="name" axisLine={false} tickLine={false} tick={{fill: '#6B7280', fontSize: 12}} dy={10} />
                        <YAxis axisLine={false} tickLine={false} tick={{fill: '#6B7280', fontSize: 12}} />
                        <Tooltip 
                            contentStyle={{ borderRadius: '8px', border: 'none', boxShadow: '0 4px 6px -1px rgba(0, 0, 0, 0.1)' }}
                            cursor={{ stroke: '#8884d8', strokeWidth: 1 }}
                        />
                        <Area type="monotone" dataKey="amount" stroke="#8884d8" fillOpacity={1} fill="url(#colorAmount)" strokeWidth={3} />
                    </AreaChart>
                </ResponsiveContainer>
             </div>
          </div>
      </div>
      )}

      {/* Filters */}
      <div className="card-custom p-3 mb-4 border-0 shadow-sm">
        <div className="d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center">
           {/* Type Filter */}
           <div className="btn-group">
              <button 
                className={`btn btn-sm px-3 ${typeFilter === '' ? 'text-white fw-bold' : 'btn-light border text-muted'}`}
                style={{ backgroundColor: typeFilter === '' ? '#ea580c' : 'transparent', borderColor: typeFilter === '' ? '#ea580c' : '#e5e7eb' }}
                onClick={() => { setTypeFilter(''); setPage(1); }}
              >
                All
              </button>
              <button 
                className={`btn btn-sm px-3 ${typeFilter === 'deposit' ? 'text-white fw-bold' : 'btn-light border text-muted'}`}
                style={{ backgroundColor: typeFilter === 'deposit' ? '#ea580c' : 'transparent', borderColor: typeFilter === 'deposit' ? '#ea580c' : '#e5e7eb' }}
                onClick={() => { setTypeFilter('deposit'); setPage(1); }}
              >
                Deposits
              </button>
              <button 
                className={`btn btn-sm px-3 ${typeFilter === 'withdraw' ? 'text-white fw-bold' : 'btn-light border text-muted'}`}
                style={{ backgroundColor: typeFilter === 'withdraw' ? '#ea580c' : 'transparent', borderColor: typeFilter === 'withdraw' ? '#ea580c' : '#e5e7eb' }}
                onClick={() => { setTypeFilter('withdraw'); setPage(1); }}
              >
                Withdrawals
              </button>
           </div>
        </div>
      </div>

      <div className="card-custom overflow-hidden border-0 shadow-sm">
        <div className="table-responsive">
          <table className="table table-hover align-middle mb-0">
            <thead className="bg-light">
              <tr>
                <th className="px-4 py-3 text-secondary small text-uppercase">ID</th>
                <th className="px-4 py-3 text-secondary small text-uppercase">User</th>
                <th className="px-4 py-3 text-secondary small text-uppercase">Type</th>
                <th className="px-4 py-3 text-secondary small text-uppercase">Description (Meta)</th>
                <th className="px-4 py-3 text-secondary small text-uppercase text-end">Amount</th>
                <th className="px-4 py-3 text-secondary small text-uppercase text-end">Date</th>
                <th className="px-4 py-3 text-secondary small text-uppercase text-end">Action</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                <tr>
                   <td colSpan={7} className="text-center py-5">
                      <div className="spinner-border text-primary" role="status">
                        <span className="visually-hidden">Loading...</span>
                      </div>
                   </td>
                </tr>
              ) : transactions.length === 0 ? (
                <tr>
                  <td colSpan={7} className="text-center py-5 text-muted">
                    No transactions found.
                  </td>
                </tr>
              ) : (
                transactions.map((item: any) => (
                  <tr key={item.id}>
                    <td className="px-4 py-3 fw-medium">#{item.id}</td>
                    <td className="px-4 py-3">
                       <div className="d-flex flex-column">
                          <span className="fw-medium text-dark">{item.user?.name || 'Unknown'}</span>
                          <span className="small text-muted">{item.user?.email}</span>
                       </div>
                    </td>
                    <td className="px-4 py-3">
                        <span className={`badge ${item.type === 'deposit' ? 'bg-success-subtle text-success' : 'bg-danger-subtle text-danger'} border px-3 py-2 rounded-pill text-uppercase`}>
                            {item.type}
                        </span>
                    </td>
                    <td className="px-4 py-3">
                        <span className="text-muted small">
                            {item.meta?.description || (item.meta ? JSON.stringify(item.meta).substring(0, 50) : '-')}
                        </span>
                    </td>
                    <td className="px-4 py-3 text-end fw-bold fs-6">
                        <span className={getAmountColor(item.type)}>
                           {item.type === 'withdraw' ? '-' : '+'}{parseFloat(item.amount).toFixed(2)}
                        </span>
                    </td>
                    <td className="px-4 py-3 text-end text-muted small">
                       {new Date(item.created_at).toLocaleString()}
                    </td>
                    <td className="px-4 py-3 text-end">
                       <Link href={`/admin/transactions/${item.id}`} className="btn btn-sm btn-light border d-inline-flex align-items-center gap-1">
                           <Eye size={14} />
                           View
                       </Link>
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
        
        {!loading && pagination && (
            <div className="p-3 border-top">
                <Pagination 
                   currentPage={page}
                   lastPage={pagination.last_page}
                   onPageChange={setPage}
                />
            </div>
        )}
      </div>
    </div>
  );
}
