"use client";

import PageHeader from "@/components/admin/PageHeader";
import { User, Bell, Lock, Moon, Shield, Key, Smartphone, Monitor } from "lucide-react";
import { useState, useEffect } from "react";
import { cn } from "@/lib/utils";
import { useAuth } from "@/context/AuthContext";
import { useAdmin } from "@/context/AdminContext";

export default function AdminSettingsPage() {
  const [activeTab, setActiveTab] = useState("profile");
  const { user } = useAuth();
  const { theme, toggleTheme } = useAdmin();

  // Local state for form fields to handling editing
  const [firstName, setFirstName] = useState("");
  const [lastName, setLastName] = useState("");
  const [email, setEmail] = useState("");
  const [bio, setBio] = useState("Super Administrator for Deligo Platform.");

  useEffect(() => {
    if (user) {
        const parts = user.name.split(" ");
        setFirstName(parts[0] || "");
        setLastName(parts.slice(1).join(" ") || "");
        setEmail(user.email || "");
    }
  }, [user]);

  const TABS = [
    { id: "profile", label: "Profile", icon: User },
    { id: "notifications", label: "Notifications", icon: Bell },
    { id: "security", label: "Security", icon: Lock },
    { id: "appearance", label: "Appearance", icon: Moon }, // Combined Appearance & Language
  ];

  const handleThemeChange = (selectedTheme: 'dark' | 'light') => {
      if (theme !== selectedTheme) {
          toggleTheme();
      }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Settings" 
        description="Manage your account preferences and system configuration."
      />

      <div className="row g-4">
        {/* Sidebar Navigation */}
        <div className="col-12 col-md-3">
          <div className="card-custom p-2 bg-sidebar sticky-top" style={{ top: '100px' }}>
             <div className="list-group list-group-flush gap-1">
               {TABS.map((tab) => (
                 <button
                    key={tab.id}
                    onClick={() => setActiveTab(tab.id)}
                    className={cn(
                      "list-group-item list-group-item-action border-0 rounded-3 d-flex align-items-center gap-2 mb-1 transition-all",
                      activeTab === tab.id 
                        ? "fw-bold text-white" 
                        : "text-muted-custom fw-semibold bg-transparent hover-bg-custom"
                    )}
                    style={{ 
                      backgroundColor: activeTab === tab.id ? 'var(--primary)' : undefined,
                      color: activeTab === tab.id ? 'white' : 'var(--muted-text)'
                    }}
                 >
                     <tab.icon size={18} /> {tab.label}
                 </button>
               ))}
             </div>
          </div>
        </div>

        {/* Content Area */}
        <div className="col-12 col-md-9">
            {/* Profile Section */}
            {activeTab === "profile" && (
                <div className="card-custom p-4 animate-fade-in">
                    <h5 className="fw-bold mb-4" style={{ color: 'var(--text-color)' }}>Profile Information</h5>
                    <div className="d-flex align-items-center gap-4 mb-5">
                        <div className="rounded-circle d-flex align-items-center justify-content-center text-white fw-bold fs-3 shadow-lg" 
                             style={{ width: '80px', height: '80px', background: 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)' }}>
                            {user?.name ? user.name.charAt(0).toUpperCase() : "A"}
                        </div>
                        <div>
                            <button className="btn btn-sm btn-outline-secondary rounded-pill px-3 fw-bold me-2" style={{ borderColor: 'var(--border-color)', color: 'var(--text-color)' }}>Change Avatar</button>
                            <button className="btn btn-sm btn-link text-danger text-decoration-none fw-bold">Remove</button>
                        </div>
                    </div>

                    <form onSubmit={(e) => e.preventDefault()}>
                        <div className="row g-3">
                            <div className="col-12 col-md-6">
                                <label className="form-label small fw-bold text-muted-custom">First Name</label>
                                <input 
                                    type="text" 
                                    className="form-control" 
                                    value={firstName}
                                    onChange={(e) => setFirstName(e.target.value)}
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                                />
                            </div>
                             <div className="col-12 col-md-6">
                                <label className="form-label small fw-bold text-muted-custom">Last Name</label>
                                <input 
                                    type="text" 
                                    className="form-control" 
                                    value={lastName}
                                    onChange={(e) => setLastName(e.target.value)}
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                                />
                            </div>
                            <div className="col-12">
                                <label className="form-label small fw-bold text-muted-custom">Email Address</label>
                                <input 
                                    type="email" 
                                    className="form-control" 
                                    value={email}
                                    onChange={(e) => setEmail(e.target.value)}
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                                />
                            </div>
                            <div className="col-12">
                                <label className="form-label small fw-bold text-muted-custom">Bio</label>
                                <textarea 
                                    className="form-control" 
                                    rows={3} 
                                    value={bio}
                                    onChange={(e) => setBio(e.target.value)}
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                                ></textarea>
                            </div>
                        </div>
                        <div className="mt-4 text-end">
                            <button className="btn btn-primary px-4 fw-bold shadow-sm">Save Changes</button>
                        </div>
                    </form>
                </div>
            )}

            {/* Notifications Section */}
            {activeTab === "notifications" && (
                <div className="card-custom p-4 animate-fade-in">
                    <h5 className="fw-bold mb-4" style={{ color: 'var(--text-color)' }}>Notification Preferences</h5>
                     
                     <div className="mb-4">
                        <h6 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                            <Bell size={18} className="text-primary-custom" /> Email Alerts
                        </h6>
                        <div className="d-flex flex-column gap-3">
                             <div className="form-check form-switch ps-0 d-flex justify-content-between align-items-center">
                                <div>
                                    <label className="form-check-label fw-semibold text-color-custom ms-2" htmlFor="check1">New Order Alerts</label>
                                    <p className="small text-muted-custom ms-2 mb-0">Get notified when a new order is placed.</p>
                                </div>
                                <input className="form-check-input ms-0" type="checkbox" id="check1" defaultChecked />
                            </div>
                             <div className="form-check form-switch ps-0 d-flex justify-content-between align-items-center">
                                <div>
                                    <label className="form-check-label fw-semibold text-color-custom ms-2" htmlFor="check2">New Vendor Signup</label>
                                    <p className="small text-muted-custom ms-2 mb-0">Get notified when a new store registers.</p>
                                </div>
                                <input className="form-check-input ms-0" type="checkbox" id="check2" defaultChecked />
                            </div>
                             <div className="form-check form-switch ps-0 d-flex justify-content-between align-items-center">
                                <div>
                                    <label className="form-check-label fw-semibold text-color-custom ms-2" htmlFor="check3">System Updates</label>
                                    <p className="small text-muted-custom ms-2 mb-0">Get notified about maintenance and updates.</p>
                                </div>
                                <input className="form-check-input ms-0" type="checkbox" id="check3" />
                            </div>
                        </div>
                     </div>
                </div>
            )}

            {/* Security Section */}
            {activeTab === "security" && (
                <div className="d-flex flex-column gap-4 animate-fade-in">
                    <div className="card-custom p-4">
                        <h5 className="fw-bold mb-4" style={{ color: 'var(--text-color)' }}>Change Password</h5>
                        <form onSubmit={(e) => e.preventDefault()}>
                             <div className="row g-3">
                                <div className="col-12">
                                    <label className="form-label small fw-bold text-muted-custom">Current Password</label>
                                    <div className="position-relative">
                                         <input type="password" className="form-control" style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                                         <Key size={16} className="position-absolute top-50 end-0 translate-middle-y me-3 text-muted-custom" />
                                    </div>
                                </div>
                                <div className="col-12 col-md-6">
                                    <label className="form-label small fw-bold text-muted-custom">New Password</label>
                                    <input type="password" className="form-control" style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                                </div>
                                 <div className="col-12 col-md-6">
                                    <label className="form-label small fw-bold text-muted-custom">Confirm Password</label>
                                    <input type="password" className="form-control" style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                                </div>
                             </div>
                             <div className="mt-4 text-end">
                                <button className="btn btn-primary px-4 fw-bold shadow-sm">Update Password</button>
                            </div>
                        </form>
                    </div>

                    <div className="card-custom p-4">
                         <h5 className="fw-bold mb-4" style={{ color: 'var(--text-color)' }}>Two-Factor Authentication</h5>
                         <div className="d-flex justify-content-between align-items-center">
                             <div className="d-flex gap-3">
                                 <div className="rounded-circle bg-success bg-opacity-10 d-flex align-items-center justify-content-center" style={{ width: '48px', height: '48px' }}>
                                     <Shield size={24} className="text-success" />
                                 </div>
                                 <div>
                                     <h6 className="fw-bold mb-1" style={{ color: 'var(--text-color)' }}>2FA is currently active</h6>
                                     <p className="small text-muted-custom mb-0">Your account is secured with Google Authenticator.</p>
                                 </div>
                             </div>
                             <button className="btn btn-outline-danger fw-bold btn-sm px-3">Disable</button>
                         </div>
                    </div>
                </div>
            )}

            {/* Appearance Section */}
             {activeTab === "appearance" && (
                <div className="card-custom p-4 animate-fade-in">
                    <h5 className="fw-bold mb-4" style={{ color: 'var(--text-color)' }}>Appearance & Language</h5>
                    
                     <div className="mb-4 pb-4 border-bottom" style={{ borderColor: 'var(--border-color)' }}>
                         <label className="form-label small fw-bold text-muted-custom mb-3">Interface Theme</label>
                         <div className="row g-3">
                             {['Light', 'Dark', 'Follow System'].map((themeName) => (
                                 <div className="col-4" key={themeName}>
                                     <label className="d-block cursor-pointer">
                                         <input 
                                            type="radio" 
                                            name="theme" 
                                            className="d-none peer" 
                                            checked={
                                                (themeName === 'Light' && theme === 'light') ||
                                                (themeName === 'Dark' && theme === 'dark') || 
                                                (themeName === 'Follow System' && false) // Not implementing 'Follow System' logic fully yet
                                            }
                                            onChange={() => {
                                                if (themeName === 'Light') handleThemeChange('light');
                                                if (themeName === 'Dark') handleThemeChange('dark');
                                            }}
                                        />
                                         <div className="p-3 rounded-3 border text-center transition-all hover-bg-custom" 
                                              style={{ 
                                                  borderColor: (themeName === 'Light' && theme === 'light') || (themeName === 'Dark' && theme === 'dark') ? 'var(--primary)' : 'var(--border-color)',
                                                  backgroundColor: 'var(--bg-color)',
                                                  boxShadow: (themeName === 'Light' && theme === 'light') || (themeName === 'Dark' && theme === 'dark') ? '0 0 0 2px var(--primary-light)' : 'none'
                                               }}
                                         >
                                             {themeName === 'Light' && <Monitor size={24} className={cn("mb-2", theme === 'light' ? "text-primary-custom" : "text-muted-custom")} />}
                                             {themeName === 'Dark' && <Moon size={24} className={cn("mb-2", theme === 'dark' ? "text-primary-custom" : "text-muted-custom")} />}
                                             {themeName === 'Follow System' && <Smartphone size={24} className="mb-2 text-muted-custom" />}
                                             <p className={cn("small fw-bold mb-0", ((themeName === 'Light' && theme === 'light') || (themeName === 'Dark' && theme === 'dark')) ? "text-primary-custom" : "text-muted-custom")}>{themeName}</p>
                                         </div>
                                     </label>
                                 </div>
                             ))}
                         </div>
                     </div>

                     <div>
                        <label className="form-label small fw-bold text-muted-custom">Language</label>
                        <select className="form-select" style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}>
                            <option>English (United States)</option>
                            <option>Arabic (العربية)</option>
                            <option>French (Français)</option>
                        </select>
                        <p className="small text-muted-custom mt-2">Adjust the language of the admin panel interface.</p>
                     </div>
                </div>
            )}
        </div>
      </div>
    </div>
  );
}

