"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import { Search, MoreHorizontal, MapPin, User, Car, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, Loader2, Clock, Calendar } from "lucide-react";
import { useState, useEffect } from "react";
import { RideService } from "@/services/api";
import toast from "react-hot-toast";

export default function AdminRidesPage() {
  const [rides, setRides] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState("all");
  const [searchTerm, setSearchTerm] = useState("");
  const [pagination, setPagination] = useState<any>(null);

  useEffect(() => {
    fetchRides(1);
  }, [filter]);

  const fetchRides = async (page = 1) => {
    setLoading(true);
    try {
      const params: any = {
        page,
        search: searchTerm || undefined,
      };

      if (filter !== "all") {
        params.status = filter;
      }

      const response = await RideService.getAll(params);
      setRides(response.data.data || []);
      setPagination(response.data.meta);
    } catch (error) {
      console.error("Failed to fetch rides:", error);
      toast.error("Failed to load rides");
    } finally {
      setLoading(false);
    }
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    fetchRides(1);
  };

  const getStatusConfig = (status: string) => {
    const s = status.toLowerCase();
    if (s === "completed" || s === "finished") return { color: "#10b981", bgColor: "rgba(16, 185, 129, 0.1)", label: "Completed" };
    if (s === "ongoing" || s === "accepted" || s === "riding" || s === "started") return { color: "#3b82f6", bgColor: "rgba(59, 130, 246, 0.1)", label: "Ongoing" };
    if (s === "cancelled" || s === "failed") return { color: "#ef4444", bgColor: "rgba(239, 68, 68, 0.1)", label: "Cancelled" };
    return { color: "#f59e0b", bgColor: "rgba(245, 158, 11, 0.1)", label: status };
  };

  return (
    <div className="py-2">
      <div className="d-flex justify-content-between align-items-end mb-4">
        <PageHeader 
          title="Ride Management" 
          description="Monitor active rides and trip history."
        />
      </div>

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3 overflow-auto" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
            {["all", "ongoing", "completed", "cancelled"].map((tab) => (
              <button
                key={tab}
                onClick={() => setFilter(tab)}
                className={`px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all ${filter === tab ? 'shadow-sm' : ''}`}
                style={{
                  fontSize: '0.85rem',
                  backgroundColor: filter === tab ? 'var(--text-color)' : 'transparent',
                  color: filter === tab ? 'var(--bg-color)' : 'var(--muted-text)',
                  whiteSpace: 'nowrap'
                }}
              >
                {tab}
              </button>
            ))}
          </div>
          
          <form onSubmit={handleSearch} className="position-relative" style={{ width: '280px' }}>
            <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
            <input 
              type="text" 
              placeholder="Search passengers or ID..." 
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="form-control ps-5 py-2"
              style={{ 
                borderRadius: '12px',
                backgroundColor: 'var(--bg-color)',
                borderColor: 'var(--border-color)',
                color: 'var(--text-color)',
                fontSize: '0.9rem'
              }}
            />
          </form>
        </div>

        {/* Table */}
        <div className="table-responsive">
          <table className="table mb-0 align-middle">
            <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
              <tr>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>ID & Date</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Passenger</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Driver</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Route</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Fare</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Status</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                <tr>
                  <td colSpan={7} className="text-center py-5">
                    <div className="d-flex flex-column align-items-center gap-2">
                       <Loader2 size={32} className="text-primary spin-animation" />
                       <span className="text-muted-custom small">Fetching rides...</span>
                    </div>
                  </td>
                </tr>
              ) : rides.length === 0 ? (
                <tr>
                   <td colSpan={7} className="text-center py-5">
                      <div className="text-muted-custom">No rides found matching your criteria.</div>
                   </td>
                </tr>
              ) : rides.map((ride) => {
                const status = getStatusConfig(ride.status);
                return (
                  <tr key={ride.id} className="hover-bg-custom transition-all" style={{ borderBottom: '1px solid var(--border-color)' }}>
                    <td className="px-4 py-3">
                       <div className="d-flex flex-column">
                          <span className="fw-bold small" style={{ color: 'var(--text-color)' }}>#{ride.id}</span>
                          <span className="extra-small text-muted-custom d-flex align-items-center gap-1">
                             <Calendar size={10} />
                             {new Date(ride.created_at).toLocaleDateString()}
                          </span>
                       </div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="d-flex align-items-center gap-2">
                        <div className="rounded-circle d-flex align-items-center justify-content-center bg-secondary bg-opacity-10 text-secondary" style={{ width: '32px', height: '32px' }}>
                            <User size={16} />
                        </div>
                        <div className="d-flex flex-column">
                           <span className="fw-bold small" style={{ color: 'var(--text-color)' }}>{ride.user?.name || "Guest"}</span>
                           <span className="extra-small text-muted-custom">{ride.user?.mobile || "N/A"}</span>
                        </div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                       <div className="d-flex align-items-center gap-2">
                        <div className="rounded-circle d-flex align-items-center justify-content-center bg-primary bg-opacity-10 text-primary" style={{ width: '32px', height: '32px' }}>
                            <Car size={16} />
                        </div>
                        <div className="d-flex flex-column">
                           <span className="small fw-semibold" style={{ color: 'var(--text-color)' }}>{ride.driver?.user?.name || "Not Assigned"}</span>
                           <span className="extra-small text-muted-custom">{ride.vehicle_type?.name || "N/A"}</span>
                        </div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="d-flex flex-column" style={{ maxWidth: '200px' }}>
                        <div className="d-flex align-items-center gap-1 text-muted-custom text-truncate small">
                           <MapPin size={12} className="text-success flex-shrink-0" /> {ride.address_from}
                        </div>
                        <div className="d-flex align-items-center gap-1 text-muted-custom ps-1 border-start ms-1 my-1 text-truncate small" style={{ borderColor: 'var(--border-color)' }}>
                          <MapPin size={12} className="text-danger flex-shrink-0" /> {ride.address_to}
                        </div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                      <span className="fw-bold text-success">${Number(ride.final_fare_total || ride.estimated_fare_total || 0).toFixed(2)}</span>
                    </td>
                    <td className="px-4 py-3">
                      <span 
                        className="badge rounded-pill fw-bold border"
                        style={{
                          backgroundColor: status.bgColor,
                          color: status.color,
                          borderColor: status.color + '22',
                          padding: '0.4em 0.8em',
                          fontSize: '0.7rem'
                        }}
                      >
                        {status.label}
                      </span>
                    </td>
                    <td className="px-4 py-3 text-end">
                      <ActionMenu 
                        onView={() => console.log('View', ride.id)}
                        onEdit={() => console.log('Edit', ride.id)}
                        onDelete={() => console.log('Delete', ride.id)}
                      />
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        {pagination && pagination.total > 0 && (
          <div className="p-4 border-top d-flex flex-column flex-md-row justify-content-between align-items-center gap-3" style={{ borderColor: 'var(--border-color)' }}>
            <div className="small text-muted-custom">
              Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> rides
            </div>
            <div className="d-flex gap-2 flex-wrap">
              <button
                onClick={() => fetchRides(1)}
                disabled={pagination.current_page === 1}
                className={`btn btn-sm border d-flex align-items-center gap-1 ${pagination.current_page === 1 ? 'btn-light disabled' : 'btn-white shadow-sm'}`}
              >
                <ChevronsLeft size={14} /> First
              </button>
              <button
                onClick={() => fetchRides(pagination.current_page - 1)}
                disabled={pagination.current_page === 1}
                className={`btn btn-sm border d-flex align-items-center gap-1 ${pagination.current_page === 1 ? 'btn-light disabled' : 'btn-white shadow-sm'}`}
              >
                <ChevronLeft size={14} /> Prev
              </button>
              
              <div className="d-none d-sm-flex align-items-center gap-1">
                {pagination.links
                  ?.filter((link: any) => !isNaN(parseInt(link.label)))
                  .slice(0, 5)
                  .map((link: any, index: number) => {
                    const isCurrent = link.active;
                    const pageNumber = parseInt(link.label);
                    return (
                      <button
                        key={index}
                        onClick={() => fetchRides(pageNumber)}
                        disabled={isCurrent}
                        className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border text-muted'}`}
                        style={{ minWidth: '32px', height: '32px' }}
                      >
                        {link.label}
                      </button>
                    );
                  })}
              </div>

              <button
                onClick={() => fetchRides(pagination.current_page + 1)}
                disabled={pagination.current_page === pagination.last_page}
                className={`btn btn-sm border d-flex align-items-center gap-1 ${pagination.current_page === pagination.last_page ? 'btn-light disabled' : 'btn-white shadow-sm'}`}
              >
                Next <ChevronRight size={14} />
              </button>
              <button
                onClick={() => fetchRides(pagination.last_page)}
                disabled={pagination.current_page === pagination.last_page}
                className={`btn btn-sm border d-flex align-items-center gap-1 ${pagination.current_page === pagination.last_page ? 'btn-light disabled' : 'btn-white shadow-sm'}`}
              >
                Last <ChevronsRight size={14} />
              </button>
            </div>
          </div>
        )}
      </div>

      <style jsx>{`
        .spin-animation {
          animation: spin 1s linear infinite;
        }
        @keyframes spin {
          from { transform: rotate(0deg); }
          to { transform: rotate(360deg); }
        }
        .extra-small {
          font-size: 0.7rem;
        }
        .hover-bg-custom:hover {
          background-color: var(--hover-bg);
        }
      `}</style>
    </div>
  );
}
