"use client";

import PageHeader from "@/components/admin/PageHeader";
import { Save, X, Upload, Store, MapPin, Tag, FileText, Globe, Plus, Trash2 } from "lucide-react";
import Link from "next/link";
import { useState, useEffect } from "react";
import { ProviderService, CategoryService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

interface Category {
  id: number;
  title: string;
  slug: string;
}

interface MetaField {
  key: string;
  value: string;
}

interface AvailabilitySlot {
  day: string;
  from: string;
  to: string;
}

export default function CreateProviderPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [categories, setCategories] = useState<Category[]>([]);
  const [subcategories, setSubcategories] = useState<Category[]>([]);
  const [bannerPreviews, setBannerPreviews] = useState<(string | null)[]>([null, null, null, null, null]);
  
  // Form State
  const [formData, setFormData] = useState({
    name_en: "",
    name_ar: "",
    details_en: "",
    details_ar: "",
    email: "",
    mobile_number: "",
    password: "",
    category_ids: [] as string[],
    subcategory_ids: [] as string[],
    address: "",
    latitude: "",
    longitude: "",
    is_verified: false,
    banners: [null, null, null, null, null] as (File | null)[]
  });

  const [metaFields, setMetaFields] = useState<MetaField[]>([
    { key: "", value: "" }
  ]);

  const [availability, setAvailability] = useState<AvailabilitySlot[]>([
    { day: "sun", from: "10:00 AM", to: "06:00 PM" },
    { day: "mon", from: "10:00 AM", to: "06:00 PM" },
    { day: "tue", from: "10:00 AM", to: "06:00 PM" },
    { day: "wed", from: "10:00 AM", to: "06:00 PM" },
    { day: "thu", from: "10:00 AM", to: "06:00 PM" },
    { day: "fri", from: "10:00 AM", to: "06:00 PM" },
    { day: "sat", from: "10:00 AM", to: "06:00 PM" }
  ]);

  useEffect(() => {
    fetchCategories();
  }, []);

  useEffect(() => {
    if (formData.category_ids.length > 0) {
      fetchSubcategories(formData.category_ids[0]);
    } else {
      setSubcategories([]);
    }
  }, [formData.category_ids]);

  const fetchCategories = async () => {
    try {
      // Fetch all categories from all pages
      let allCategories: any[] = [];
      let currentPage = 1;
      let lastPage = 1;
      
      do {
        const response = await CategoryService.getAll({ page: currentPage });
        const data = response.data.data || response.data || [];
        allCategories = [...allCategories, ...data];
        
        const meta = response.data.meta;
        if (meta) {
          lastPage = meta.last_page || 1;
        }
        
        currentPage++;
      } while (currentPage <= lastPage);
      
      // Filter to show only parent categories (parent_id = null)
      const parentCategories = allCategories.filter((cat: any) => !cat.parent_id);
      setCategories(parentCategories);
    } catch (err) {
      console.error("Failed to fetch categories", err);
      toast.error("Failed to load categories");
    }
  };

  const fetchSubcategories = async (categoryId: string) => {
    try {
      // Use the dedicated endpoint with categories parameter
      const response = await CategoryService.getAll({ 
        categories: categoryId,
        pagination: 0 
      });
      
      const subcategoriesData = response.data || [];
      setSubcategories(subcategoriesData);
    } catch (err) {
      console.error("Failed to fetch subcategories", err);
      setSubcategories([]);
    }
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value, type } = e.target;
    
    if (type === 'checkbox') {
        setFormData(prev => ({
            ...prev,
            [name]: (e.target as HTMLInputElement).checked
        }));
    } else {
        setFormData(prev => ({
            ...prev,
            [name]: value
        }));
    }
  };

  const handleBannerChange = (index: number, e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      const newBanners = [...formData.banners];
      newBanners[index] = file;
      setFormData(prev => ({ ...prev, banners: newBanners }));
      
      const newPreviews = [...bannerPreviews];
      newPreviews[index] = URL.createObjectURL(file);
      setBannerPreviews(newPreviews);
    }
  };

  const handleMetaChange = (index: number, field: 'key' | 'value', value: string) => {
    const newMeta = [...metaFields];
    newMeta[index][field] = value;
    setMetaFields(newMeta);
  };

  const addMetaField = () => {
    setMetaFields([...metaFields, { key: "", value: "" }]);
  };

  const removeMetaField = (index: number) => {
    setMetaFields(metaFields.filter((_, i) => i !== index));
  };

  const handleAvailabilityChange = (index: number, field: 'from' | 'to', value: string) => {
    const newAvailability = [...availability];
    newAvailability[index][field] = value;
    setAvailability(newAvailability);
  };

  const addAvailability = () => {
    setAvailability([...availability, { day: "sun", from: "10:00 AM", to: "06:00 PM" }]);
  };

  const removeAvailability = (index: number) => {
    setAvailability(availability.filter((_, i) => i !== index));
  };

  const handleCategoryToggle = (categoryId: string) => {
    setFormData(prev => ({
      ...prev,
      category_ids: prev.category_ids.includes(categoryId)
        ? prev.category_ids.filter(id => id !== categoryId)
        : [...prev.category_ids, categoryId]
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      // Build translation objects
      const nameObj: Record<string, string> = {};
      if (formData.name_en) nameObj.en = formData.name_en;
      if (formData.name_ar) nameObj.ar = formData.name_ar;
      
      const detailsObj: Record<string, string> = {};
      if (formData.details_en) detailsObj.en = formData.details_en;
      if (formData.details_ar) detailsObj.ar = formData.details_ar;

      // Build meta object
      const metaObject = metaFields.reduce((acc, field) => {
        if (field.key) acc[field.key] = field.value;
        return acc;
      }, {} as Record<string, string>);

      // Create translations arrays
      const nameTranslations = [];
      if (formData.name_en) nameTranslations.push({ language: "en", text: formData.name_en });
      if (formData.name_ar) nameTranslations.push({ language: "ar", text: formData.name_ar });
      
      const detailsTranslations = [];
      if (formData.details_en) detailsTranslations.push({ language: "en", text: formData.details_en });
      if (formData.details_ar) detailsTranslations.push({ language: "ar", text: formData.details_ar });

      const providerData: any = {
        name_translations: nameTranslations,
        details_translations: detailsTranslations.length > 0 ? detailsTranslations : undefined,
        _name: nameObj,
        _details: Object.keys(detailsObj).length > 0 ? detailsObj : undefined,
        email: formData.email,
        mobile_number: formData.mobile_number,
        password: formData.password,
        categories: formData.category_ids,
        subcategories: formData.subcategory_ids,
        address: formData.address,
        is_verified: formData.is_verified,
        availability: availability
      };

      // Add optional lat/long
      if (formData.latitude && formData.latitude.toString().trim() !== "") {
        providerData.latitude = parseFloat(formData.latitude.toString());
      }
      if (formData.longitude && formData.longitude.toString().trim() !== "") {
        providerData.longitude = parseFloat(formData.longitude.toString());
      }

      // Add meta as JSON string
      if (Object.keys(metaObject).length > 0) {
        providerData.meta = JSON.stringify(metaObject);
      }

      const axios = (await import('axios')).default;
      const API_URL = process.env.NEXT_PUBLIC_API_URL || 'https://admin.captaincity.net/api';
      const token = localStorage.getItem('token');
      
      // Send as JSON
      const response = await axios.post(`${API_URL}/admin/providers`, providerData, {
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
          'Authorization': token ? `Bearer ${token}` : '',
        },
      });
      
      // Upload banners if exists
      if (response.data?.data?.id) {
        const providerId = response.data.data.id;
        for (let i = 0; i < formData.banners.length; i++) {
          if (formData.banners[i]) {
            const bannerData = new FormData();
            bannerData.append(`banner_${i}`, formData.banners[i]!);
            bannerData.append("_method", "PUT");
            await axios.post(`${API_URL}/admin/providers/${providerId}`, bannerData, {
              headers: {
                'Authorization': token ? `Bearer ${token}` : '',
              },
            });
          }
        }
      }

      toast.success("Provider created successfully!");
      router.push("/admin/providers");
    } catch (err: any) {
      console.error(err);
      toast.error(err.response?.data?.message || "Failed to create provider");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Create New Provider" 
        description="Add a new service provider with detailed information."
      />

      <div className="row justify-content-center">
        <div className="col-12">
          <form onSubmit={handleSubmit}>
            <div className="row g-4">
              
              {/* Main Info Column */}
              <div className="col-12 col-lg-8">
                {/* User Information */}
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <Globe size={20} className="text-primary-custom" />
                    User Information
                  </h5>
                  
                  <div className="row g-3">
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Email</label>
                      <input 
                        type="email" 
                        name="email"
                        value={formData.email}
                        onChange={handleChange}
                        className="form-control" 
                        required
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Mobile Number</label>
                      <input 
                        type="text" 
                        name="mobile_number"
                        value={formData.mobile_number}
                        onChange={handleChange}
                        className="form-control" 
                        required
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Password</label>
                      <input 
                        type="password" 
                        name="password"
                        value={formData.password}
                        onChange={handleChange}
                        className="form-control" 
                        required
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                  </div>
                </div>

                {/* Name Translation */}
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <Globe size={20} className="text-primary-custom" />
                    Name Translation
                  </h5>
                  
                  <div className="row g-3">
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Name English</label>
                      <input 
                        type="text" 
                        name="name_en"
                        value={formData.name_en}
                        onChange={handleChange}
                        className="form-control" 
                        required
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Name Arabic</label>
                      <input 
                        type="text" 
                        name="name_ar"
                        value={formData.name_ar}
                        onChange={handleChange}
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                  </div>
                </div>

                {/* Details Translation */}
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <FileText size={20} className="text-primary-custom" />
                    Details Translation
                  </h5>
                  
                  <div className="row g-3">
                    <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Details English</label>
                      <textarea 
                        name="details_en"
                        value={formData.details_en}
                        onChange={handleChange}
                        className="form-control" 
                        rows={3}
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Details Arabic</label>
                      <textarea 
                        name="details_ar"
                        value={formData.details_ar}
                        onChange={handleChange}
                        className="form-control" 
                        rows={3}
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                  </div>
                </div>

                {/* Verification */}
                <div className="card-custom p-3 mb-3">
                  <div className="form-check form-switch">
                    <input 
                      className="form-check-input" 
                      type="checkbox" 
                      role="switch" 
                      id="is_verified"
                      name="is_verified"
                      checked={formData.is_verified}
                      onChange={handleChange}
                      style={{ 
                        width: '3em', 
                        height: '1.5em', 
                        cursor: 'pointer',
                        backgroundColor: formData.is_verified ? '#10b981' : 'var(--border-color)',
                        borderColor: formData.is_verified ? '#10b981' : 'var(--border-color)' 
                      }} 
                    />
                    <label className="form-check-label fw-bold text-muted-custom cursor-pointer ms-2" htmlFor="is_verified">
                      Is Verified?
                    </label>
                  </div>
                </div>

                {/* Categories */}
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <Tag size={20} className="text-primary-custom" />
                    Categories
                  </h5>
                  <select 
                    className="form-select"
                    value={formData.category_ids[0] || ""}
                    onChange={(e) => {
                      const categoryId = e.target.value;
                      setFormData(prev => ({
                        ...prev,
                        category_ids: categoryId ? [categoryId] : [],
                        subcategory_ids: [] // Reset subcategories when category changes
                      }));
                    }}
                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                  >
                    <option value="">Select Category...</option>
                    {categories.map(cat => (
                      <option key={cat.id} value={cat.id.toString()}>
                        {cat.title}
                      </option>
                    ))}
                  </select>
                </div>

                {/* Subcategories */}
                {formData.category_ids.length > 0 && (
                  <div className="card-custom p-3 mb-3">
                    <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                      <Tag size={20} className="text-primary-custom" />
                      Subcategories
                    </h5>
                    <div className="d-flex flex-column gap-2">
                      {subcategories.length === 0 ? (
                        <p className="text-muted small">No subcategories available for this category</p>
                      ) : (
                        subcategories.map(subcat => (
                          <div key={subcat.id} className="form-check">
                            <input 
                              className="form-check-input" 
                              type="checkbox" 
                              id={`subcat-${subcat.id}`}
                              checked={formData.subcategory_ids.includes(subcat.id.toString())}
                              onChange={(e) => {
                                const subcatId = subcat.id.toString();
                                setFormData(prev => ({
                                  ...prev,
                                  subcategory_ids: e.target.checked 
                                    ? [...prev.subcategory_ids, subcatId]
                                    : prev.subcategory_ids.filter(id => id !== subcatId)
                                }));
                              }}
                            />
                            <label className="form-check-label" htmlFor={`subcat-${subcat.id}`}>
                              {subcat.title}
                            </label>
                          </div>
                        ))
                      )}
                    </div>
                  </div>
                )}

                {/* Address & Location */}
                <div className="card-custom p-3 mb-3">
                   <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <MapPin size={20} className="text-primary-custom" />
                    Address & Location
                  </h5>

                  <div className="row g-3">
                     <div className="col-12">
                      <label className="form-label small fw-bold text-muted-custom">Address</label>
                      <input 
                        type="text" 
                        name="address"
                        value={formData.address}
                        onChange={handleChange}
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Latitude</label>
                      <input 
                        type="number" 
                        step="any"
                        name="latitude"
                        value={formData.latitude}
                        onChange={handleChange}
                        placeholder="e.g., 30.540917"
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                     <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Longitude</label>
                      <input 
                        type="number" 
                        step="any"
                        name="longitude"
                        value={formData.longitude}
                        onChange={handleChange}
                        placeholder="e.g., 31.366250"
                        className="form-control" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                      />
                    </div>
                  </div>
                </div>

                {/* Banners */}
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <Upload size={20} className="text-primary-custom" />
                    Banners
                  </h5>
                  <div className="row g-3">
                    {[0, 1, 2, 3, 4].map(index => (
                      <div key={index} className="col-12 col-md-6">
                        <label className="form-label small fw-bold text-muted-custom">Banner {index + 1}</label>
                        <input 
                          type="file" 
                          accept="image/*"
                          onChange={(e) => handleBannerChange(index, e)}
                          className="form-control"
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        />
                        {bannerPreviews[index] && (
                          <img src={bannerPreviews[index]!} alt={`Banner ${index + 1}`} className="mt-2 rounded" style={{ maxHeight: '100px', width: 'auto' }} />
                        )}
                      </div>
                    ))}
                  </div>
                </div>

                {/* Availability */}
                <div className="card-custom p-3 mb-3">
                  <div className="d-flex justify-content-between align-items-center mb-3">
                    <h5 className="fw-bold mb-0 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                      <FileText size={20} className="text-primary-custom" />
                      Availability
                    </h5>
                    <button type="button" onClick={addAvailability} className="btn btn-sm btn-outline-primary">
                      <Plus size={16} /> Add
                    </button>
                  </div>
                  
                  {availability.map((slot, index) => (
                    <div key={index} className="row g-2 mb-2 align-items-center">
                      <div className="col-3">
                        <label className="small fw-bold text-muted-custom">Day</label>
                        <select 
                          className="form-select form-select-sm"
                          value={slot.day}
                          onChange={(e) => {
                            const newAvailability = [...availability];
                            newAvailability[index].day = e.target.value;
                            setAvailability(newAvailability);
                          }}
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        >
                          <option value="sun">Sunday</option>
                          <option value="mon">Monday</option>
                          <option value="tue">Tuesday</option>
                          <option value="wed">Wednesday</option>
                          <option value="thu">Thursday</option>
                          <option value="fri">Friday</option>
                          <option value="sat">Saturday</option>
                        </select>
                      </div>
                      <div className="col-4">
                        <label className="small fw-bold text-muted-custom">From</label>
                        <input 
                          type="text"
                          className="form-control form-control-sm"
                          value={slot.from}
                          onChange={(e) => handleAvailabilityChange(index, 'from', e.target.value)}
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        />
                      </div>
                      <div className="col-4">
                        <label className="small fw-bold text-muted-custom">To</label>
                        <input 
                          type="text"
                          className="form-control form-control-sm"
                          value={slot.to}
                          onChange={(e) => handleAvailabilityChange(index, 'to', e.target.value)}
                          style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                        />
                      </div>
                      <div className="col-1 d-flex align-items-end pb-1">
                        <button type="button" onClick={() => removeAvailability(index)} className="btn btn-sm text-danger p-0">
                          <Trash2 size={18} />
                        </button>
                      </div>
                    </div>
                  ))}
                </div>

                 {/* Meta Fields */}
                 <div className="card-custom p-3">
                   <div className="d-flex justify-content-between align-items-center mb-3">
                     <h5 className="fw-bold mb-0 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                        <FileText size={20} className="text-primary-custom" />
                        Meta Fields
                     </h5>
                     <button type="button" onClick={addMetaField} className="btn btn-sm btn-outline-primary">
                        <Plus size={16} /> Add Meta
                     </button>
                   </div>
                   
                   <div className="d-flex flex-column gap-2">
                     {metaFields.map((field, index) => (
                        <div key={index} className="row g-2 align-items-center">
                            <div className="col-5">
                                <label className="small fw-bold text-muted-custom mb-1">Key</label>
                                <input 
                                    type="text" 
                                    className="form-control form-control-sm"
                                    value={field.key}
                                    onChange={(e) => handleMetaChange(index, 'key', e.target.value)}
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                                />
                            </div>
                             <div className="col-6">
                                <label className="small fw-bold text-muted-custom mb-1">Value</label>
                                <input 
                                    type="text" 
                                    className="form-control form-control-sm"
                                    value={field.value}
                                    onChange={(e) => handleMetaChange(index, 'value', e.target.value)}
                                    style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}
                                />
                            </div>
                            <div className="col-1 d-flex align-items-end justify-content-center pb-1">
                                <button type="button" onClick={() => removeMetaField(index)} className="btn btn-sm text-danger p-0 border-0">
                                    <X size={18} />
                                </button>
                            </div>
                        </div>
                     ))}
                   </div>
                </div>
              </div>

              {/* Sidebar Column */}
              <div className="col-12 col-lg-4">
                <div className="card-custom p-4">
                  <p className="text-muted small">Complete all required fields to create a new provider.</p>
                </div>
              </div>

              {/* Actions Footer */}
              <div className="col-12">
                 <div className="d-flex gap-3 mt-2">
                    <button 
                      type="submit"
                      disabled={loading}
                      className="btn btn-success d-flex align-items-center gap-2 px-4 fw-bold shadow-sm"
                      style={{ background: 'linear-gradient(135deg, rgb(249, 115, 22) 0%, rgb(234, 88, 12) 100%)', border: 'none' }}
                    >
                      <Save size={18} />
                      {loading ? "SAVING..." : "SAVE"}
                    </button>
                    <Link 
                      href="/admin/providers"
                      className="btn btn-danger d-flex align-items-center gap-2 fw-semibold px-4"
                      style={{ background: '#ef4444', border: 'none' }}
                    >
                      <X size={18} />
                      BACK
                    </Link>
                  </div>
              </div>

            </div>
          </form>
        </div>
      </div>
    </div>
  );
}
