"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, Package, DollarSign, Layers, AlertCircle, RefreshCw, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight } from "lucide-react";
import { useState, useEffect } from "react";
import { ProductService } from "@/services/api";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";

export default function AdminProductsPage() {
  const router = useRouter();
  const [products, setProducts] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState("all");
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedProduct, setSelectedProduct] = useState<any | null>(null);
  const [pagination, setPagination] = useState<any>(null);

  useEffect(() => {
    fetchProducts(1);
  }, [filter]);

  const fetchProducts = async (page = 1) => {
    try {
      setLoading(true);
      const params: any = { page };
      if (filter !== "all") params.status = filter;
      
      const response = await ProductService.getAll(params);
      setProducts(response.data.data || []);
      setPagination(response.data.meta);
    } catch (err) {
      console.error("Failed to fetch products:", err);
      toast.error("Failed to load products");
    } finally {
      setLoading(false);
    }
  };

  /* 
  const filteredProducts = products.filter(product => {
      const status = getProductStatus(product);
      if (filter === "all") return true;
      return status === filter;
  });
  */
  const filteredProducts = products;

  const handleDeleteClick = (product: any) => {
    setSelectedProduct(product);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedProduct) return;
    
    try {
        await ProductService.delete(selectedProduct.id);
        toast.success("Product deleted successfully");
        fetchProducts(); // Refresh list
    } catch (err) {
        console.error("Failed to delete product:", err);
        toast.error("Failed to delete product");
    } finally {
        setDeleteModalOpen(false);
        setSelectedProduct(null);
    }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Product Management" 
        description="Manage products, inventory, and pricing across all stores."
        actionLabel="Add New Product"
        actionLink="/admin/products/create"
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
            {["all", "in_stock", "low_stock", "out_of_stock"].map((tab) => (
              <button
                key={tab}
                onClick={() => setFilter(tab)}
                className={`px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all text-nowrap ${filter === tab ? 'shadow-sm' : ''}`}
                style={{
                  fontSize: '0.85rem',
                  backgroundColor: filter === tab ? 'var(--text-color)' : 'transparent',
                  color: filter === tab ? 'var(--bg-color)' : 'var(--muted-text)',
                }}
              >
                {tab.split('_').join(' ')}
              </button>
            ))}
          </div>
          
          <div className="d-flex gap-2 w-100 w-md-auto">
             <div className="position-relative flex-grow-1" style={{ width: '280px' }}>
                <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
                <input 
                type="text" 
                placeholder="Search products..." 
                className="form-control ps-5 py-2"
                style={{ 
                    borderRadius: '12px',
                    backgroundColor: 'var(--bg-color)',
                    borderColor: 'var(--border-color)',
                    color: 'var(--text-color)',
                    fontSize: '0.9rem'
                }}
                />
            </div>
            <button 
                onClick={() => fetchProducts(pagination?.current_page || 1)} 
                className="btn btn-light border d-flex align-items-center justify-content-center"
                style={{ width: '40px', height: '40px', borderColor: 'var(--border-color)' }}
                title="Refresh List"
            >
                <RefreshCw size={18} className={loading ? "animate-spin" : ""} />
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          <table className="table mb-0 align-middle">
            <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
              <tr>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Product</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Category</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Details</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Price</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Stock</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Status</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Type</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                  <tr>
                      <td colSpan={7} className="text-center py-5">
                          <div className="spinner-border text-primary" role="status">
                            <span className="visually-hidden">Loading...</span>
                          </div>
                      </td>
                  </tr>
              ) : filteredProducts.length === 0 ? (
                  <tr>
                      <td colSpan={7} className="text-center py-5 text-muted">
                          No products found.
                      </td>
                  </tr>
              ) : (
                filteredProducts.map((product) => {
                    // Construct full image URL with better error handling
                    let imageUrl = product.mediaurls?.images?.[0]?.default || product.image;
                    
                    // If imageUrl exists and doesn't start with http, prepend API base URL
                    if (imageUrl && !imageUrl.startsWith('http')) {
                        const API_BASE = process.env.NEXT_PUBLIC_API_URL || 'https://admin.captaincity.net/api';
                        const baseUrl = API_BASE.replace('/api', '');
                        imageUrl = `${baseUrl}${imageUrl.startsWith('/') ? '' : '/'}${imageUrl}`;
                    }
                    
                    // Debug logging for images
                    if (imageUrl) {
                        console.log(`Product ${product.id} image URL:`, imageUrl);
                    }
                    
                    return (
                        <tr key={product.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                        <td className="px-4 py-3">
                            <div className="d-flex align-items-center gap-3">
                            <div className="rounded-3 d-flex align-items-center justify-content-center overflow-hidden position-relative" style={{ 
                                width: '50px', 
                                height: '50px', 
                                backgroundColor: 'var(--hover-bg)',
                                color: 'var(--muted-text)',
                                border: '1px solid var(--border-color)'
                            }}>
                                {imageUrl ? (
                                    <>
                                        <img 
                                            src={imageUrl} 
                                            alt={product.title} 
                                            className="w-100 h-100 object-fit-cover position-absolute top-0 start-0" 
                                            onError={(e) => {
                                                console.error(`Failed to load image for product ${product.id}:`, imageUrl);
                                                (e.target as HTMLImageElement).style.display = 'none';
                                                const parent = (e.target as HTMLImageElement).parentElement;
                                                const fallback = parent?.querySelector('.fallback-icon');
                                                if (fallback) {
                                                    fallback.classList.remove('d-none');
                                                }
                                            }}
                                            onLoad={() => {
                                                console.log(`Successfully loaded image for product ${product.id}`);
                                            }}
                                        />
                                        <Package size={24} className="fallback-icon d-none" />
                                    </>
                                ) : (
                                    <Package size={24} />
                                )}
                            </div>
                            <div>
                                <p className="fw-bold mb-0" style={{ color: 'var(--text-color)' }}>{product.title}</p>
                                <p className="small mb-0 text-muted-custom">ID: #{product.id}</p>
                            </div>
                            </div>
                        </td>
                        <td className="px-4 py-3">
                            <div className="d-flex align-items-center gap-2">
                                <Layers size={14} className="text-primary-custom" />
                                <span className="small fw-medium" style={{ color: 'var(--muted-text)' }}>
                                    {product.categories?.[0]?.title || "Uncategorized"}
                                </span>
                            </div>
                        </td>
                        <td className="px-4 py-3">
                            <span className="small text-muted-custom d-block" style={{ maxWidth: '200px' }}>
                                {product.detail || "No details available"}
                            </span>
                        </td>
                        <td className="px-4 py-3">
                            <span className="fw-bold text-success d-flex align-items-center gap-1">
                            <DollarSign size={14} />
                            {parseFloat(product.price).toFixed(2)}
                            </span>
                        </td>
                        <td className="px-4 py-3">
                            <span className={`fw-bold small ${(product.stock || 0) <= 0 ? 'text-danger' : (product.stock || 0) < 10 ? 'text-warning' : 'text-muted-custom'}`}>
                            {product.stock || 0} units
                            </span>
                        </td>
                        <td className="px-4 py-3">
                            <span 
                            className="badge rounded-pill fw-bold border"
                            style={{
                                backgroundColor: (product.stock || 0) > 0 ? 'rgba(16, 185, 129, 0.1)' : 'rgba(239, 68, 68, 0.1)',
                                color: (product.stock || 0) > 0 ? '#10b981' : '#ef4444',
                                borderColor: (product.stock || 0) > 0 ? 'rgba(16, 185, 129, 0.2)' : 'rgba(239, 68, 68, 0.2)',
                                padding: '0.5em 0.8em'
                            }}
                            >
                            {(product.stock || 0) > 0 ? "In Stock" : "Out of Stock"}
                            </span>
                        </td>
                        <td className="px-4 py-3">
                            <span className={`badge rounded-pill fw-bold border ${product.meta?.food_type === 'veg' ? 'bg-success-subtle text-success border-success-subtle' : 'bg-danger-subtle text-danger border-danger-subtle'}`}>
                                {product.meta?.food_type || "Standard"}
                            </span>
                        </td>
                        <td className="px-4 py-3 text-end">
                            <ActionMenu 
                                onView={() => router.push(`/admin/products/${product.id}`)}
                                onEdit={() => router.push(`/admin/products/${product.id}/edit`)}
                                onDelete={() => handleDeleteClick(product)}
                            />
                        </td>
                        </tr>
                    );
                })
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Pagination */}
      {pagination && (
        <div className="d-flex justify-content-between align-items-center mt-4">
          <div className="small text-muted-custom">
            Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> results
          </div>
          <div className="d-flex gap-2 flex-wrap">
            {/* First Page Button */}
            <button
              onClick={() => fetchProducts(1)}
              disabled={pagination.current_page === 1}
              className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
              style={{ padding: '0.5rem 1rem' }}
              title="Go to first page"
            >
              <ChevronsLeft size={16} /> First
            </button>
            
            {/* Previous Button */}
            <button
              onClick={() => fetchProducts(pagination.current_page - 1)}
              disabled={pagination.current_page === 1}
              className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
              style={{ padding: '0.5rem 1rem' }}
            >
              <ChevronLeft size={16} /> Previous
            </button>
            
            {/* Page Numbers */}
            <div className="d-flex align-items-center gap-1 flex-wrap">
              {pagination.links
                ?.filter((link: any) => {
                  // Only show links with numeric labels (actual page numbers)
                  const pageNum = parseInt(link.label);
                  return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                })
                .map((link: any, index: number) => {
                  const isCurrent = link.active;
                  const pageNumber = parseInt(link.label);
                  
                  return (
                    <button
                      key={index}
                      onClick={() => fetchProducts(pageNumber)}
                      disabled={isCurrent}
                      className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border'}`}
                      style={{ minWidth: '32px', height: '32px', padding: '0 0.5rem' }}
                    >
                      {link.label}
                    </button>
                  );
                })}
            </div>
            
            {/* Next Button */}
            <button
              onClick={() => fetchProducts(pagination.current_page + 1)}
              disabled={pagination.current_page === pagination.last_page}
              className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
              style={{ padding: '0.5rem 1rem' }}
            >
              Next <ChevronRight size={16} />
            </button>
            
            {/* Last Page Button */}
            <button
              onClick={() => fetchProducts(pagination.last_page)}
              disabled={pagination.current_page === pagination.last_page}
              className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
              style={{ padding: '0.5rem 1rem' }}
              title="Go to last page"
            >
              Last <ChevronsRight size={16} />
            </button>
          </div>
        </div>
      )}
      
       <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={selectedProduct?.title}
        title="Delete Product"
        description="Are you sure you want to delete this product? This action cannot be undone."
      />
    </div>
  );
}
