"use client";

import PageHeader from "@/components/admin/PageHeader";
import { Save, X, Upload, Globe, Plus, Trash2, FileText, ShoppingBag, Layers, DollarSign, Archive, MoreHorizontal, ChevronDown, ChevronUp } from "lucide-react";
import Link from "next/link";
import { useState, useEffect } from "react";
import { StoreService, CategoryService, ProductService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

interface Category {
  id: number;
  title: string;
}

interface Vendor {
  id: number;
  name: string;
}

interface MetaField {
  key: string;
  value: string;
}

interface AddonChoice {
  title_en: string;
  title_ar: string;
  price: string;
}

interface AddonGroup {
  title_en: string;
  title_ar: string;
  min_choices: string;
  max_choices: string;
  choices: AddonChoice[];
}

export default function CreateProductPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  const [categories, setCategories] = useState<Category[]>([]);
  const [vendors, setVendors] = useState<Vendor[]>([]);
  const [imagePreview, setImagePreview] = useState<string | null>(null);

  // Form State
  const [formData, setFormData] = useState({
    title_en: "",
    title_ar: "",
    detail_en: "",
    detail_ar: "",
    vendor_id: "",
    categories: [] as string[], // Multi-select support
    price: "",
    stock_quantity: "",
    image: null as File | null
  });

  const [addonGroups, setAddonGroups] = useState<AddonGroup[]>([]);
  const [metaFields, setMetaFields] = useState<MetaField[]>([]);

  useEffect(() => {
    fetchDependencies();
  }, []);

  const fetchDependencies = async () => {
    try {
      const [catRes, vendorRes] = await Promise.all([
        CategoryService.getAll(),
        StoreService.getAll()
      ]);
      setCategories(catRes.data.data || catRes.data || []);
      setVendors(vendorRes.data.data || vendorRes.data || []);
    } catch (err) {
      console.error("Failed to load dependencies", err);
      toast.error("Failed to load categories or vendors");
    }
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleCategoryChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
      const selectedOptions = Array.from(e.target.selectedOptions, option => option.value);
      setFormData(prev => ({ ...prev, categories: selectedOptions }));
  };

  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setFormData(prev => ({ ...prev, image: file }));
      setImagePreview(URL.createObjectURL(file));
    }
  };

  // --- Meta Fields Logic ---
  const handleMetaChange = (index: number, field: 'key' | 'value', value: string) => {
    const newMeta = [...metaFields];
    newMeta[index][field] = value;
    setMetaFields(newMeta);
  };
  const addMetaField = () => setMetaFields([...metaFields, { key: "", value: "" }]);
  const removeMetaField = (index: number) => setMetaFields(metaFields.filter((_, i) => i !== index));

  // --- Addon Group Logic ---
  const addAddonGroup = () => {
    setAddonGroups([...addonGroups, {
      title_en: "", title_ar: "", min_choices: "0", max_choices: "1", choices: []
    }]);
  };
  const removeAddonGroup = (index: number) => {
    setAddonGroups(addonGroups.filter((_, i) => i !== index));
  };
  const updateAddonGroup = (index: number, field: keyof AddonGroup, value: string) => {
    const newGroups = [...addonGroups];
    // @ts-ignore
    newGroups[index][field] = value;
    setAddonGroups(newGroups);
  };
  
  // --- Addon Choice Logic ---
  const addChoice = (groupIndex: number) => {
    const newGroups = [...addonGroups];
    newGroups[groupIndex].choices.push({ title_en: "", title_ar: "", price: "0" });
    setAddonGroups(newGroups);
  };
  const removeChoice = (groupIndex: number, choiceIndex: number) => {
      const newGroups = [...addonGroups];
      newGroups[groupIndex].choices = newGroups[groupIndex].choices.filter((_, i) => i !== choiceIndex);
      setAddonGroups(newGroups);
  };
  const updateChoice = (groupIndex: number, choiceIndex: number, field: keyof AddonChoice, value: string) => {
      const newGroups = [...addonGroups];
      // @ts-ignore
      newGroups[groupIndex].choices[choiceIndex][field] = value;
      setAddonGroups(newGroups);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
        // Construct Payload
        const productData: any = {
            // Basic
            price: formData.price,
            stock: formData.stock_quantity, // Assuming backend uses 'stock'
            vendor_id: formData.vendor_id,
            categories: formData.categories,
            
            // Translations (Format depends on backend, assuming array for create)
            title_translations: [
                { language: 'en', text: formData.title_en },
                { language: 'ar', text: formData.title_ar }
            ],
            detail_translations: [
                { language: 'en', text: formData.detail_en },
                { language: 'ar', text: formData.detail_ar }
            ],

            // Addons
            addon_groups: addonGroups.map(group => ({
                title_translations: [
                    { language: 'en', text: group.title_en },
                    { language: 'ar', text: group.title_ar }
                ],
                min_choices: group.min_choices,
                max_choices: group.max_choices,
                choices: group.choices.map(choice => ({
                    title_translations: [
                        { language: 'en', text: choice.title_en },
                        { language: 'ar', text: choice.title_ar }
                    ],
                    price: choice.price
                }))
            })),

            // Meta
            meta: JSON.stringify(
                metaFields.reduce((acc, field) => {
                    if (field.key) acc[field.key] = field.value;
                    return acc;
                }, {} as Record<string, string>)
            )
        };

        // 1. Create Product
        const response = await ProductService.create(productData);
        
        // 2. Upload Image
        if (formData.image && response.data?.data?.id) {
             const axios = (await import('axios')).default;
             const API_URL = process.env.NEXT_PUBLIC_API_URL || 'https://admin.captaincity.net/api';
             const token = localStorage.getItem('token');
             
             const imageData = new FormData();
             imageData.append("image", formData.image);
             imageData.append("_method", "PUT");
             
             await axios.post(`${API_URL}/admin/products/${response.data.data.id}`, imageData, {
                headers: { 'Authorization': token ? `Bearer ${token}` : '' }
             });
        }

        toast.success("Product created successfully!");
        router.push("/admin/products");

    } catch (err: any) {
        console.error(err);
        toast.error(err.response?.data?.message || "Failed to create product");
    } finally {
        setLoading(false);
    }
  };

  return (
    <div className="py-2">
      <PageHeader title="Create New Product" description="Add a new product to the catalog." />

      <div className="row justify-content-center">
        <div className="col-12">
          <form onSubmit={handleSubmit}>
            <div className="row g-4">
              
              {/* LEFT COLUMN: Main Content */}
              <div className="col-12 col-lg-8">
                
                {/* General Information (Combined) */}
                <div className="card-custom p-3 mb-3">
                  <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                    <FileText size={20} className="text-primary-custom" />
                    General Information
                  </h5>
                  
                  {/* Title Section */}
                  <div className="mb-4">
                     <h6 className="small fw-bold text-muted mb-2 border-bottom pb-1">Product Title</h6>
                     <div className="row g-3">
                      <div className="col-12 col-md-6">
                        <label className="form-label small fw-bold text-muted-custom">Title (English)</label>
                        <input type="text" name="title_en" value={formData.title_en} onChange={handleChange} className="form-control" placeholder="Product Name" required style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                      </div>
                      <div className="col-12 col-md-6">
                           <label className="form-label small fw-bold text-muted-custom">Title (Arabic)</label>
                           <input type="text" name="title_ar" value={formData.title_ar} onChange={handleChange} className="form-control" placeholder="اسم المنتج" required style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                      </div>
                    </div>
                  </div>

                  {/* Detail Section */}
                  <div>
                     <h6 className="small fw-bold text-muted mb-2 border-bottom pb-1">Description / Details</h6>
                     <div className="row g-3">
                      <div className="col-12 col-md-6">
                         <label className="form-label small fw-bold text-muted-custom">Detail (English)</label>
                         <textarea name="detail_en" value={formData.detail_en} onChange={handleChange} className="form-control" rows={4} placeholder="Product Description..." required style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                      </div>
                      <div className="col-12 col-md-6">
                         <label className="form-label small fw-bold text-muted-custom">Detail (Arabic)</label>
                         <textarea name="detail_ar" value={formData.detail_ar} onChange={handleChange} className="form-control" rows={4} placeholder="وصف المنتج..." required style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                      </div>
                    </div>
                  </div>
                </div>

                {/* Addon Groups */}
                <div className="card-custom p-3 mb-3">
                    <div className="d-flex justify-content-between align-items-center mb-3">
                        <h5 className="fw-bold mb-0 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                            <Layers size={20} className="text-primary-custom" />
                            Addon Groups
                        </h5>
                        <button type="button" onClick={addAddonGroup} className="btn btn-sm btn-outline-primary fw-bold" style={{ borderColor: 'var(--border-color)', color: 'var(--text-color)' }}>
                            <Plus size={16} /> Add Group
                        </button>
                    </div>

                    {addonGroups.length === 0 && (
                        <div className="d-flex flex-column align-items-center justify-content-center py-5 border rounded-3 border-dashed" style={{ borderColor: 'var(--border-color)' }}>
                           <Layers size={32} className="text-muted mb-2 opacity-50" />
                           <p className="small text-muted mb-0">No addon groups configured.</p>
                           <button type="button" onClick={addAddonGroup} className="btn btn-link btn-sm fw-bold">Click to add a group</button>
                        </div>
                    )}

                    {addonGroups.map((group, groupIndex) => (
                        <div key={groupIndex} className="card-custom p-4 mb-3 border position-relative" style={{ backgroundColor: 'var(--bg-color)' }}>
                            <div className="d-flex justify-content-between align-items-center mb-3 pb-2 border-bottom">
                                <h6 className="fw-bold m-0 text-primary-custom d-flex align-items-center gap-2">
                                  <span className="badge bg-primary text-white rounded-circle d-flex align-items-center justify-content-center" style={{ width: '24px', height: '24px', fontSize: '0.75rem' }}>{groupIndex + 1}</span>
                                  Addon Group
                                </h6>
                                <button type="button" onClick={() => removeAddonGroup(groupIndex)} className="btn btn-sm btn-outline-danger d-flex align-items-center gap-1 border-0 fw-bold" style={{ fontSize: '0.8rem' }}>
                                  <Trash2 size={16} /> Delete Group
                                </button>
                            </div>
                            
                            {/* Group Fields */}
                            <div className="row g-3 mb-4">
                                <div className="col-12 col-md-6">
                                    <label className="form-label small fw-bold text-muted-custom">Group Title (English)</label>
                                    <input placeholder="e.g. Size, Toppings" value={group.title_en} onChange={e => updateAddonGroup(groupIndex, 'title_en', e.target.value)} className="form-control" style={{ backgroundColor: 'var(--hover-bg)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }} />
                                </div>
                                <div className="col-12 col-md-6">
                                    <label className="form-label small fw-bold text-muted-custom">Group Title (Arabic)</label>
                                    <input placeholder="عنوان المجموعة" value={group.title_ar} onChange={e => updateAddonGroup(groupIndex, 'title_ar', e.target.value)} className="form-control" style={{ backgroundColor: 'var(--hover-bg)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }} />
                                </div>
                                <div className="col-6 col-md-3">
                                    <label className="form-label small fw-bold text-muted-custom">Min Choices</label>
                                    <input type="number" min="0" value={group.min_choices} onChange={e => updateAddonGroup(groupIndex, 'min_choices', e.target.value)} className="form-control" style={{ backgroundColor: 'var(--hover-bg)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }} />
                                </div>
                                <div className="col-6 col-md-3">
                                    <label className="form-label small fw-bold text-muted-custom">Max Choices</label>
                                    <input type="number" min="1" value={group.max_choices} onChange={e => updateAddonGroup(groupIndex, 'max_choices', e.target.value)} className="form-control" style={{ backgroundColor: 'var(--hover-bg)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }} />
                                </div>
                            </div>

                            {/* Choices */}
                            <div className="bg-light rounded-3 p-3 border" style={{ backgroundColor: 'var(--hover-bg)', borderColor: 'var(--border-color)' }}>
                                <div className="d-flex justify-content-between align-items-center mb-3">
                                     <span className="small fw-bold" style={{ color: 'var(--text-color)' }}>Choices / Options</span>
                                     <button type="button" onClick={() => addChoice(groupIndex)} className="btn btn-sm btn-info text-white fw-bold d-flex align-items-center gap-1 shadow-sm" style={{ fontSize: '0.8rem' }}>
                                       <Plus size={14} /> Add Choice
                                     </button>
                                </div>
                                
                                {group.choices.length === 0 ? (
                                  <div className="text-center py-3 text-muted small border rounded-2 border-dashed bg-white opacity-75">
                                    No choices added yet. Click "Add Choice" to start.
                                  </div>
                                ) : (
                                  <div className="d-flex flex-column gap-2">
                                    {group.choices.map((choice, choiceIndex) => (
                                      <div key={choiceIndex} className="row g-2 align-items-center bg-white p-2 rounded-2 border shadow-sm">
                                           <div className="col-12 col-md-4">
                                              <input placeholder="Choice (EN)" value={choice.title_en} onChange={e => updateChoice(groupIndex, choiceIndex, 'title_en', e.target.value)} className="form-control form-control-sm border-0 bg-light" />
                                           </div>
                                           <div className="col-12 col-md-4">
                                              <input placeholder="Choice (AR)" value={choice.title_ar} onChange={e => updateChoice(groupIndex, choiceIndex, 'title_ar', e.target.value)} className="form-control form-control-sm border-0 bg-light" />
                                           </div>
                                            <div className="col-8 col-md-3">
                                              <div className="input-group input-group-sm">
                                                <span className="input-group-text border-0 bg-light text-muted">$</span>
                                                <input type="number" placeholder="0.00" value={choice.price} onChange={e => updateChoice(groupIndex, choiceIndex, 'price', e.target.value)} className="form-control border-0 bg-light" />
                                              </div>
                                           </div>
                                           <div className="col-4 col-md-1 text-end">
                                               <button type="button" onClick={() => removeChoice(groupIndex, choiceIndex)} className="btn btn-sm text-danger hover-bg-light rounded-circle p-1">
                                                 <Trash2 size={16} />
                                               </button>
                                           </div>
                                      </div>
                                    ))}
                                  </div>
                                )}
                            </div>
                        </div>
                    ))}
                </div>

                 {/* Meta Fields (Advanced) */}
                 <div className="card-custom p-3">
                   <div className="d-flex justify-content-between align-items-center mb-3">
                     <h5 className="fw-bold mb-0 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                        <MoreHorizontal size={20} className="text-primary-custom" />
                        Meta Fields (Advanced)
                     </h5>
                     <button type="button" onClick={addMetaField} className="btn btn-sm btn-outline-primary fw-bold" style={{ borderColor: 'var(--border-color)', color: 'var(--text-color)' }}>
                        <Plus size={16} /> Add Meta
                     </button>
                   </div>
                   
                   {/* Meta List (Same as Stores) */}
                    <div className="d-flex flex-column gap-2">
                     {metaFields.map((field, index) => (
                        <div key={index} className="row g-2 align-items-center">
                            <div className="col-5">
                                <label className="small fw-bold text-muted-custom mb-1">Key</label>
                                <input type="text" className="form-control form-control-sm" value={field.key} onChange={(e) => handleMetaChange(index, 'key', e.target.value)} style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                            </div>
                             <div className="col-6">
                                <label className="small fw-bold text-muted-custom mb-1">Value</label>
                                <input type="text" className="form-control form-control-sm" value={field.value} onChange={(e) => handleMetaChange(index, 'value', e.target.value)} style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                            </div>
                            <div className="col-1 d-flex align-items-end justify-content-center pb-1">
                                <button type="button" onClick={() => removeMetaField(index)} className="btn btn-sm text-danger p-0 border-0"><X size={18} /></button>
                            </div>
                        </div>
                     ))}
                   </div>
                 </div>

              </div>

              {/* RIGHT COLUMN: Sidebar (Media, Inventory, Actions) */}
              <div className="col-12 col-lg-4">
                 
                 {/* 1. Media First */}
                 <div className="card-custom p-3 mb-3">
                   <h5 className="fw-bold mb-3" style={{ color: 'var(--text-color)' }}>Product Image</h5>
                   <div className="d-flex align-items-center justify-content-center p-3 rounded-3 border-2 border-dashed transition-all hover-bg-custom position-relative overflow-hidden" 
                      style={{ borderColor: 'var(--border-color)', backgroundColor: 'var(--bg-color)', cursor: 'pointer', minHeight: '220px' }}>
                      <input type="file" accept="image/*" onChange={handleImageChange} className="position-absolute w-100 h-100 opacity-0 cursor-pointer" style={{ zIndex: 10 }} />
                      {imagePreview ? (
                          <div className="w-100 h-100 position-absolute top-0 start-0">
                              <img src={imagePreview} alt="Preview" className="w-100 h-100 object-fit-cover" />
                              <div className="position-absolute bottom-0 start-0 w-100 p-2 bg-dark bg-opacity-50 text-white text-center small">
                                  Click to change
                              </div>
                          </div>
                      ) : (
                        <div className="text-center">
                            <div className="mb-3 d-inline-flex p-3 rounded-circle shadow-sm" style={{ backgroundColor: 'var(--hover-bg)' }}>
                                <Upload size={24} className="text-primary-custom" />
                            </div>
                            <p className="small fw-bold mb-1" style={{ color: 'var(--text-color)' }}>Drop image here</p>
                            <p className="small text-muted-custom mb-0">or click to upload</p>
                        </div>
                      )}
                   </div>
                 </div>

                 {/* 2. Organization & Inventory */}
                 <div className="card-custom p-3 mb-3">
                    <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                         <ShoppingBag size={20} className="text-primary-custom" />
                         Organization
                    </h5>
                    
                    <div className="mb-3">
                        <label className="form-label small fw-bold text-muted-custom">Vendor</label>
                        <select name="vendor_id" value={formData.vendor_id} onChange={handleChange} className="form-select" required style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }}>
                            <option value="">Select Vendor...</option>
                            {vendors.map(v => <option key={v.id} value={v.id}>{v.name}</option>)}
                        </select>
                    </div>
                    
                    <div className="mb-4">
                        <label className="form-label small fw-bold text-muted-custom">Categories</label>
                        <select name="categories" multiple value={formData.categories} onChange={handleCategoryChange} className="form-select" required style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)', height: '120px' }}>
                            {categories.map(c => <option key={c.id} value={c.id}>{c.title}</option>)}
                        </select>
                        <small className="text-muted d-block mt-1" style={{ fontSize: '0.7rem' }}>Hold Ctrl/Cmd to select multiple</small>
                    </div>

                    <div className="border-top pt-3 mt-3">
                         <h6 className="small fw-bold text-muted mb-3">Pricing & Inventory</h6>
                         <div className="mb-3">
                            <label className="form-label small fw-bold text-muted-custom">Price</label>
                            <div className="input-group">
                                 <span className="input-group-text" style={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }}><DollarSign size={16} /></span>
                                <input type="number" name="price" value={formData.price} onChange={handleChange} className="form-control" placeholder="0.00" required style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                            </div>
                        </div>

                        <div className="mb-0">
                            <label className="form-label small fw-bold text-muted-custom">Stock Quantity</label>
                             <div className="input-group">
                                 <span className="input-group-text" style={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }}><Archive size={16} /></span>
                                <input type="number" name="stock_quantity" value={formData.stock_quantity} onChange={handleChange} className="form-control" placeholder="0" required style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} />
                            </div>
                        </div>
                    </div>
                 </div>

              </div>

               {/* Actions Footer */}
               <div className="col-12">
                  <div className="d-flex gap-2">
                     <button type="submit" disabled={loading} className="btn btn-success fw-bold  shadow-sm py-2" style={{ background: 'linear-gradient(135deg, rgb(249, 115, 22) 0%, rgb(234, 88, 12) 100%)', border: 'none' }}>
                       {loading ? "SAVING..." : "PUBLISH PRODUCT"}
                     </button>
                     <Link href="/admin/products" className="btn btn-light fw-bold px-3 border" style={{ color: 'var(--text-color)' }}>
                       Cancel
                     </Link>
                   </div>
               </div>

            </div>
          </form>
        </div>
      </div>
    </div>
  );
}
