"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, Clock, CheckCircle, Truck, Package, MapPin, User, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight } from "lucide-react";
import { useState, useEffect } from "react";
import { OrderService } from "@/services/api";

const STATUS_CONFIG = {
  pending: { label: "Pending", color: "#f59e0b", bgColor: "rgba(245, 158, 11, 0.1)", borderColor: "rgba(245, 158, 11, 0.2)", icon: Clock },
  preparing: { label: "Preparing", color: "#3b82f6", bgColor: "rgba(59, 130, 246, 0.1)", borderColor: "rgba(59, 130, 246, 0.2)", icon: Package },
  delivering: { label: "Delivering", color: "#a855f7", bgColor: "rgba(168, 85, 247, 0.1)", borderColor: "rgba(168, 85, 247, 0.2)", icon: Truck },
  completed: { label: "Completed", color: "#10b981", bgColor: "rgba(16, 185, 129, 0.1)", borderColor: "rgba(16, 185, 129, 0.2)", icon: CheckCircle },
  cancelled: { label: "Cancelled", color: "#ef4444", bgColor: "rgba(239, 68, 68, 0.1)", borderColor: "rgba(239, 68, 68, 0.2)", icon: CheckCircle },
};

export default function AdminOrdersPage() {
  const [orders, setOrders] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState("all");
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedOrder, setSelectedOrder] = useState<any>(null);
  const [pagination, setPagination] = useState<any>(null);
  const [searchTerm, setSearchTerm] = useState("");

  useEffect(() => {
    fetchOrders(1);
  }, [filter]);

  const fetchOrders = async (page = 1) => {
    setLoading(true);
    try {
      const params: any = { 
        page,
        search: searchTerm || undefined 
      };
      
      if (filter !== "all") {
        params.status = filter;
      }

      const response = await OrderService.getAll(params);
      setOrders(response.data.data || []);
      setPagination(response.data.meta);
    } catch (error) {
      console.error("Failed to fetch orders:", error);
    } finally {
      setLoading(false);
    }
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    fetchOrders(1);
  };

  const updateStatus = async (orderId: string, status: string) => {
    try {
      await OrderService.updateStatus(orderId, status);
      fetchOrders(pagination?.current_page || 1); // Refresh current page
    } catch (error) {
      console.error("Failed to update status:", error);
    }
  };

  const handleDeleteClick = (order: any) => {
    setSelectedOrder(order);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedOrder) return;
    try {
      await OrderService.delete(selectedOrder.id);
      fetchOrders(pagination?.current_page || 1);
      setDeleteModalOpen(false);
      setSelectedOrder(null);
    } catch (error) {
      console.error("Delete failed", error);
    }
  };

  return (
    <div className="py-2">
      <div className="d-flex justify-content-between align-items-end mb-4">
        <PageHeader 
          title="Order Management" 
          description="Track and manage all orders in real-time."
        />
        <div className="d-none d-md-block">
            <div className="card-custom px-4 py-2 d-flex flex-column align-items-center justify-content-center" style={{ minWidth: '140px' }}>
                <span className="extra-small fw-bold text-uppercase text-muted-custom" style={{ letterSpacing: '1px', fontSize: '10px' }}>Active Orders</span>
                <p className="h4 fw-bold mb-0 text-primary-custom">
                    {pagination?.total || 0}
                </p>
            </div>
        </div>
      </div>

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3 overflow-auto" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
            {["all", "pending", "preparing", "delivering", "completed", "cancelled"].map((tab) => (
              <button
                key={tab}
                onClick={() => setFilter(tab)}
                className={`px-3 py-2 rounded-2 text-capitalize fw-bold border-0 transition-all ${filter === tab ? 'shadow-sm' : ''}`}
                style={{
                  fontSize: '0.85rem',
                  backgroundColor: filter === tab ? 'var(--text-color)' : 'transparent',
                  color: filter === tab ? 'var(--bg-color)' : 'var(--muted-text)',
                  whiteSpace: 'nowrap'
                }}
              >
                {tab}
              </button>
            ))}
          </div>
          
          <form onSubmit={handleSearch} className="position-relative" style={{ width: '280px' }}>
            <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
            <input 
              type="text" 
              placeholder="Search by ID or Customer..." 
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="form-control ps-5 py-2"
              style={{ 
                borderRadius: '12px',
                backgroundColor: 'var(--bg-color)',
                borderColor: 'var(--border-color)',
                color: 'var(--text-color)',
                fontSize: '0.9rem'
              }}
            />
          </form>
        </div>

        {/* Orders List */}
        <div className="p-4 d-flex flex-column gap-3">
          {loading ? (
             <div className="text-center py-5">
                <div className="spinner-border text-primary mb-3" role="status"></div>
                <p className="text-muted-custom small">Loading orders...</p>
             </div>
          ) : (
            <>
              {orders.length === 0 && (
                <div className="text-center py-5 d-flex flex-column align-items-center justify-content-center">
                    <div className="mb-3 d-inline-flex p-4 rounded-circle bg-light border">
                        <Package size={48} className="text-muted opacity-50" />
                    </div>
                    <p className="fw-bold mb-1" style={{ color: 'var(--text-color)' }}>No orders found.</p>
                    <p className="text-muted-custom small px-5" style={{ maxWidth: '300px' }}>Try adjusting your search filters to find what you're looking for.</p>
                </div>
              )}
              
              {orders.map((order) => {
                const statusConfig = STATUS_CONFIG[order.status as keyof typeof STATUS_CONFIG] || STATUS_CONFIG.pending;
                const StatusIcon = statusConfig.icon;
                
                return (
                  <div key={order.id} className="p-4 rounded-4 transition-all hover-shadow-custom border" 
                    style={{ 
                        backgroundColor: 'var(--hover-bg)', 
                        borderColor: 'var(--border-color)'
                    }}
                  >
                    <div className="d-flex flex-wrap align-items-start justify-content-between mb-4">
                      <div>
                        <div className="d-flex align-items-center gap-3 mb-2">
                          <h5 className="fw-bold mb-0 text-color-custom">Order #{order.id}</h5>
                          <span 
                            className="badge rounded-pill d-flex align-items-center gap-1 border"
                            style={{
                                backgroundColor: statusConfig.bgColor,
                                color: statusConfig.color,
                                borderColor: statusConfig.borderColor,
                                padding: '0.5em 0.8em',
                                fontSize: '0.75rem'
                            }}
                          >
                            <StatusIcon size={12} />
                            {statusConfig.label}
                          </span>
                        </div>
                        <p className="small text-muted-custom mb-0 d-flex align-items-center gap-2">
                            <Clock size={14} />
                            {new Date(order.created_at || Date.now()).toLocaleString()}
                        </p>
                      </div>
                      <ActionMenu 
                          onView={() => console.log(`View Order #${order.id}`)}
                          onEdit={() => console.log(`Edit Order #${order.id}`)}
                          onDelete={() => handleDeleteClick(order)}
                      />
                    </div>

                    <div className="row g-4 mb-4">
                      <div className="col-12 col-sm-6 col-md-3">
                        <div className="d-flex align-items-center gap-3">
                            <div className="rounded-3 d-flex align-items-center justify-content-center" style={{ width: '40px', height: '40px', backgroundColor: 'rgba(59, 130, 246, 0.1)', color: '#3b82f6' }}>
                               <User size={18} />
                            </div>
                            <div>
                               <p className="extra-small text-muted-custom mb-0 text-uppercase fw-bold" style={{ fontSize: '0.7rem' }}>Customer</p>
                               <p className="small fw-bold mb-0 text-color-custom text-truncate" style={{ maxWidth: '120px' }}>{order.user?.name || "Guest"}</p>
                            </div>
                        </div>
                      </div>
                       <div className="col-12 col-sm-6 col-md-3">
                        <div className="d-flex align-items-center gap-3">
                            <div className="rounded-3 d-flex align-items-center justify-content-center" style={{ width: '40px', height: '40px', backgroundColor: 'rgba(245, 158, 11, 0.1)', color: '#f59e0b' }}>
                               <Package size={18} />
                            </div>
                            <div>
                               <p className="extra-small text-muted-custom mb-0 text-uppercase fw-bold" style={{ fontSize: '0.7rem' }}>Store</p>
                               <p className="small fw-bold mb-0 text-color-custom text-truncate" style={{ maxWidth: '120px' }}>{order.vendor?.name || order.store?.name || "Unknown Store"}</p>
                            </div>
                        </div>
                      </div>
                       <div className="col-12 col-sm-6 col-md-3" >
                        <div className="d-flex align-items-center gap-3">
                             <div className="rounded-3 d-flex align-items-center justify-content-center fw-bold small" style={{ width: '40px', height: '40px', backgroundColor: 'rgba(16, 185, 129, 0.1)', color: '#10b981' }}>
                               ${Math.floor(order.total_amount || order.total || 0)}
                            </div>
                            <div>
                               <p className="extra-small text-muted-custom mb-0 text-uppercase fw-bold" style={{ fontSize: '0.7rem' }}>Total</p>
                               <p className="small fw-bold mb-0 text-color-custom">{order.items_count || order.items?.length || 0} Items</p>
                            </div>
                        </div>
                      </div>
                       <div className="col-12 col-sm-6 col-md-3">
                        <div className="d-flex align-items-center gap-3">
                            <div className="rounded-3 d-flex align-items-center justify-content-center" style={{ width: '40px', height: '40px', backgroundColor: 'rgba(168, 85, 247, 0.1)', color: '#a855f7' }}>
                               <MapPin size={18} />
                            </div>
                            <div>
                               <p className="extra-small text-muted-custom mb-0 text-uppercase fw-bold" style={{ fontSize: '0.7rem' }}>Address</p>
                               <p className="small fw-bold mb-0 text-color-custom text-truncate" style={{ maxWidth: '120px' }}>{order.address?.address || order.address || "N/A"}</p>
                            </div>
                        </div>
                      </div>
                    </div>

                    {order.status !== "completed" && order.status !== "cancelled" && (
                      <div className="pt-3 border-top d-flex gap-3 flex-wrap" style={{ borderColor: 'var(--border-color)' }}>
                        {order.status === "pending" && (
                          <button 
                            onClick={() => updateStatus(order.id, "preparing")}
                            className="btn btn-sm fw-bold flex-grow-1" style={{ backgroundColor: 'rgba(59, 130, 246, 0.1)', color: '#3b82f6', border: '1px solid rgba(59, 130, 246, 0.2)' }}>
                            Mark as Preparing
                          </button>
                        )}
                        {order.status === "preparing" && (
                          <button 
                            onClick={() => updateStatus(order.id, "delivering")}
                            className="btn btn-sm fw-bold flex-grow-1" style={{ backgroundColor: 'rgba(168, 85, 247, 0.1)', color: '#a855f7', border: '1px solid rgba(168, 85, 247, 0.2)' }}>
                            Assign Driver / Deliver
                          </button>
                        )}
                        {order.status === "delivering" && (
                          <button 
                             onClick={() => updateStatus(order.id, "completed")}
                            className="btn btn-sm fw-bold flex-grow-1" style={{ backgroundColor: 'rgba(16, 185, 129, 0.1)', color: '#10b981', border: '1px solid rgba(16, 185, 129, 0.2)' }}>
                            Mark as Delivered
                          </button>
                        )}
                        <button 
                            onClick={() => console.log(`View Details #${order.id}`)}
                            className="btn btn-sm flex-grow-1 fw-bold" style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', border: '1px solid var(--border-color)' }}>
                          View Details
                        </button>
                      </div>
                    )}
                  </div>
                );
              })}

              {/* Pagination */}
              {pagination && pagination.total > 0 && (
                <div className="d-flex justify-content-between align-items-center mt-4">
                  <div className="small text-muted-custom">
                    Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> orders
                  </div>
                  <div className="d-flex gap-2 flex-wrap">
                    <button
                      onClick={() => fetchOrders(1)}
                      disabled={pagination.current_page === 1}
                      className={`btn btn-sm border d-flex align-items-center gap-1 ${pagination.current_page === 1 ? 'btn-light disabled' : 'btn-white shadow-sm'}`}
                    >
                      <ChevronsLeft size={14} /> First
                    </button>
                    <button
                      onClick={() => fetchOrders(pagination.current_page - 1)}
                      disabled={pagination.current_page === 1}
                      className={`btn btn-sm border d-flex align-items-center gap-1 ${pagination.current_page === 1 ? 'btn-light disabled' : 'btn-white shadow-sm'}`}
                    >
                      <ChevronLeft size={14} /> Prev
                    </button>
                    <div className="d-flex align-items-center gap-1 flex-wrap">
                      {pagination.links
                        ?.filter((link: any) => {
                          const pageNum = parseInt(link.label);
                          return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                        })
                        .map((link: any, index: number) => {
                          const isCurrent = link.active;
                          const pageNumber = parseInt(link.label);
                          return (
                            <button
                              key={index}
                              onClick={() => fetchOrders(pageNumber)}
                              disabled={isCurrent}
                              className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border text-muted'}`}
                              style={{ minWidth: '32px', height: '32px' }}
                            >
                              {link.label}
                            </button>
                          );
                        })}
                    </div>
                    <button
                      onClick={() => fetchOrders(pagination.current_page + 1)}
                      disabled={pagination.current_page === pagination.last_page}
                      className={`btn btn-sm border d-flex align-items-center gap-1 ${pagination.current_page === pagination.last_page ? 'btn-light disabled' : 'btn-white shadow-sm'}`}
                    >
                      Next <ChevronRight size={14} />
                    </button>
                    <button
                      onClick={() => fetchOrders(pagination.last_page)}
                      disabled={pagination.current_page === pagination.last_page}
                      className={`btn btn-sm border d-flex align-items-center gap-1 ${pagination.current_page === pagination.last_page ? 'btn-light disabled' : 'btn-white shadow-sm'}`}
                    >
                      Last <ChevronsRight size={14} />
                    </button>
                  </div>
                </div>
              )}
            </>
          )}
        </div>
      </div>
      
       <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        title="Delete Order"
        itemName={selectedOrder ? `Order #${selectedOrder.id}` : undefined}
      />
    </div>
  );
}
