"use client";

import PageHeader from "@/components/admin/PageHeader";
import { FileText, DollarSign, MapPin, Loader2 } from "lucide-react";
import { useState } from "react";
import { LocationChargeService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";
import Link from "next/link";

export default function CreateLocationChargePage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);
  
  const [formData, setFormData] = useState({
    title_en: "",
    title_ar: "",
    latitude: "",
    longitude: "",
    radius: "",
    amount: ""
  });

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
      const { name, value } = e.target;
      setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const payload = {
        title: JSON.stringify({
          en: formData.title_en,
          ar: formData.title_ar
        }),
        latitude: parseFloat(formData.latitude),
        longitude: parseFloat(formData.longitude),
        radius: parseFloat(formData.radius),
        amount: parseFloat(formData.amount)
      };

      await LocationChargeService.create(payload);
      
      toast.success("Location charge created successfully");
      router.push("/admin/location-charges");
    } catch (error: any) {
      console.error("Failed to create location charge:", error);
      toast.error(error.response?.data?.message || "Failed to create location charge");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Create Location Charge" 
        description="Add a new location-based surcharge."
      />

      <div className="row justify-content-center">
        <div className="col-12">
           <form onSubmit={handleSubmit}>
            
              {/* General Information */}
              <div className="card-custom p-3 mb-3">
                <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                  <FileText size={20} className="text-primary-custom" />
                  Location Details
                </h5>

                <div className="row g-3">
                   <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Title (English)</label>
                      <input 
                        type="text" 
                        name="title_en"
                        value={formData.title_en} 
                        onChange={handleChange} 
                        className="form-control" 
                        placeholder="e.g. Airport Surcharge" 
                        required 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                      />
                   </div>
                   <div className="col-12 col-md-6">
                      <label className="form-label small fw-bold text-muted-custom">Title (Arabic)</label>
                      <input 
                        type="text" 
                        name="title_ar"
                        value={formData.title_ar} 
                        onChange={handleChange} 
                        className="form-control text-end" 
                        placeholder="رسوم المطار" 
                        style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                      />
                   </div>
                </div>
              </div>

              {/* Location Configuration */}
              <div className="card-custom p-3 mb-3">
                <h5 className="fw-bold mb-3 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                  <MapPin size={20} className="text-primary-custom" />
                  Geographic Configuration
                </h5>
                
                <div className="row g-3">
                    <div className="col-md-6">
                       <label className="form-label small fw-bold text-muted-custom">Latitude</label>
                       <input 
                         type="number" 
                         name="latitude"
                         value={formData.latitude}
                         onChange={handleChange}
                         className="form-control"
                         placeholder="24.7136"
                         step="0.0000001"
                         required
                         style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                       />
                    </div>
                    <div className="col-md-6">
                       <label className="form-label small fw-bold text-muted-custom">Longitude</label>
                       <input 
                         type="number" 
                         name="longitude"
                         value={formData.longitude}
                         onChange={handleChange}
                         className="form-control"
                         placeholder="46.6753"
                         step="0.0000001"
                         required
                         style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                       />
                    </div>
                    <div className="col-md-6">
                       <label className="form-label small fw-bold text-muted-custom">Radius (km)</label>
                       <input 
                         type="number" 
                         name="radius"
                         value={formData.radius}
                         onChange={handleChange}
                         className="form-control"
                         placeholder="5.00"
                         step="0.01"
                         min="0"
                         required
                         style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                       />
                       <small className="text-muted d-block mt-1" style={{ fontSize: '0.75rem' }}>Coverage radius in kilometers</small>
                    </div>
                    <div className="col-md-6">
                       <label className="form-label small fw-bold text-muted-custom">Charge Amount</label>
                       <div className="input-group">
                          <span className="input-group-text" style={{ backgroundColor: 'var(--bg-color)', borderColor: 'var(--border-color)', color: 'var(--text-color)' }}><DollarSign size={16} /></span>
                          <input 
                            type="number" 
                            name="amount"
                            value={formData.amount}
                            onChange={handleChange}
                            className="form-control"
                            placeholder="10.00"
                            step="0.01"
                            min="0"
                            required
                            style={{ backgroundColor: 'var(--bg-color)', color: 'var(--text-color)', borderColor: 'var(--border-color)' }} 
                          />
                       </div>
                    </div>
                </div>
              </div>

              {/* Actions */}
              <div className="col-12">
                 <div className="d-flex gap-2">
                    <button type="submit" disabled={loading} className="btn btn-success fw-bold shadow-sm py-2" style={{ background: 'linear-gradient(135deg, rgb(249, 115, 22) 0%, rgb(234, 88, 12) 100%)', border: 'none' }}>
                      {loading ? <Loader2 size={18} className="spin-animation mx-auto" /> : "CREATE LOCATION CHARGE"}
                    </button>
                    <Link href="/admin/location-charges" className="btn btn-light fw-bold px-3 border" style={{ color: 'var(--text-color)' }}>
                      Cancel
                    </Link>
                 </div>
              </div>
           </form>
        </div>
      </div>
    </div>
  );
}
