"use client";

import { useState } from "react";
import { Search, Wallet, TrendingUp, TrendingDown, ArrowUpRight, ArrowDownRight, DollarSign, User, Store, Car } from "lucide-react";
import { cn } from "@/lib/utils";

const WALLETS = [
  { id: 1, owner: "John Doe", type: "customer", balance: 125.50, transactions: 23, lastActivity: "2 hours ago" },
  { id: 2, owner: "Burger Queen", type: "vendor", balance: 4520.80, transactions: 156, lastActivity: "15 mins ago" },
  { id: 3, owner: "Mike Driver", type: "driver", balance: 890.30, transactions: 67, lastActivity: "1 hour ago" },
  { id: 4, owner: "Fresh Mart", type: "vendor", balance: 3240.00, transactions: 98, lastActivity: "30 mins ago" },
  { id: 5, owner: "Sarah Smith", type: "customer", balance: 45.20, transactions: 12, lastActivity: "3 days ago" },
];

const TRANSACTIONS = [
  { id: 1, type: "earning", amount: 45.50, source: "Order #12345", user: "Burger Queen", time: "15 mins ago" },
  { id: 2, type: "payout", amount: -500.00, source: "Bank Transfer", user: "Burger Queen", time: "1 hour ago" },
  { id: 3, type: "earning", amount: 12.50, source: "Ride #789", user: "Mike Driver", time: "2 hours ago" },
  { id: 4, type: "deposit", amount: 100.00, source: "Wallet Top-up", user: "John Doe", time: "3 hours ago" },
  { id: 5, type: "earning", amount: 67.80, source: "Order #12342", user: "Fresh Mart", time: "4 hours ago" },
];

const TYPE_CONFIG = {
  customer: { icon: User, color: "text-blue-500", bg: "bg-blue-500/10" },
  vendor: { icon: Store, color: "text-amber-500", bg: "bg-amber-500/10" },
  driver: { icon: Car, color: "text-purple-500", bg: "bg-purple-500/10" },
};

export default function AdminWalletsPage() {
  const [filter, setFilter] = useState("all");

  const filteredWallets = filter === "all" 
    ? WALLETS 
    : WALLETS.filter(w => w.type === filter);

  const totalBalance = WALLETS.reduce((sum, w) => sum + w.balance, 0);

  return (
    <div className="space-y-8">
      <div className="flex justify-between items-end">
        <div>
          <h2 className="text-3xl font-bold tracking-tight">Wallet Management</h2>
          <p className="text-white/50 mt-2">Monitor user balances and transaction history.</p>
        </div>
      </div>

      {/* Summary Cards */}
      <div className="grid md:grid-cols-4 gap-6">
        <div className="glass p-6 rounded-3xl">
          <div className="flex items-center gap-3 mb-4">
            <div className="w-12 h-12 rounded-2xl bg-emerald-500/10 flex items-center justify-center">
              <DollarSign className="w-6 h-6 text-emerald-500" />
            </div>
          </div>
          <h3 className="text-2xl font-bold">${totalBalance.toFixed(2)}</h3>
          <p className="text-sm text-white/40">Total Platform Balance</p>
        </div>
        
        {Object.entries(TYPE_CONFIG).map(([type, config]) => {
          const typeWallets = WALLETS.filter(w => w.type === type);
          const typeBalance = typeWallets.reduce((sum, w) => sum + w.balance, 0);
          const Icon = config.icon;
          
          return (
            <div key={type} className="glass p-6 rounded-3xl">
              <div className="flex items-center gap-3 mb-4">
                <div className={cn("w-12 h-12 rounded-2xl flex items-center justify-center", config.bg)}>
                  <Icon className={cn("w-6 h-6", config.color)} />
                </div>
              </div>
              <h3 className="text-2xl font-bold">${typeBalance.toFixed(2)}</h3>
              <p className="text-sm text-white/40 capitalize">{type} Wallets</p>
            </div>
          );
        })}
      </div>

      <div className="grid lg:grid-cols-2 gap-8">
        {/* Wallets List */}
        <div className="glass rounded-[32px] overflow-hidden">
          <div className="p-6 border-b border-white/5 bg-white/5">
            <h3 className="text-lg font-bold mb-4">All Wallets</h3>
            <div className="flex bg-black/20 p-1 rounded-xl">
              {["all", "customer", "vendor", "driver"].map((tab) => (
                <button
                  key={tab}
                  onClick={() => setFilter(tab)}
                  className={cn(
                    "flex-1 px-3 py-2 rounded-lg text-sm font-bold capitalize transition-all",
                    filter === tab ? "bg-white/10 text-white shadow-lg" : "text-white/40 hover:text-white"
                  )}
                >
                  {tab}
                </button>
              ))}
            </div>
          </div>

          <div className="p-6 space-y-4 max-h-[600px] overflow-y-auto">
            {filteredWallets.map((wallet) => {
              const config = TYPE_CONFIG[wallet.type as keyof typeof TYPE_CONFIG];
              const Icon = config.icon;
              
              return (
                <div key={wallet.id} className="bg-white/5 border border-white/10 rounded-2xl p-4 hover:border-primary/50 transition-all">
                  <div className="flex items-center justify-between mb-3">
                    <div className="flex items-center gap-3">
                      <div className={cn("w-10 h-10 rounded-xl flex items-center justify-center", config.bg)}>
                        <Icon className={cn("w-5 h-5", config.color)} />
                      </div>
                      <div>
                        <p className="font-bold">{wallet.owner}</p>
                        <p className="text-xs text-white/40 capitalize">{wallet.type}</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <p className="text-lg font-bold text-emerald-500">${wallet.balance.toFixed(2)}</p>
                      <p className="text-xs text-white/40">{wallet.transactions} txns</p>
                    </div>
                  </div>
                  <p className="text-xs text-white/30">Last activity: {wallet.lastActivity}</p>
                </div>
              );
            })}
          </div>
        </div>

        {/* Recent Transactions */}
        <div className="glass rounded-[32px] overflow-hidden">
          <div className="p-6 border-b border-white/5 bg-white/5">
            <h3 className="text-lg font-bold">Recent Transactions</h3>
          </div>

          <div className="p-6 space-y-4 max-h-[600px] overflow-y-auto">
            {TRANSACTIONS.map((txn) => (
              <div key={txn.id} className="flex items-center justify-between p-4 bg-white/5 rounded-2xl hover:bg-white/10 transition-colors">
                <div className="flex items-center gap-4">
                  <div className={cn(
                    "w-10 h-10 rounded-xl flex items-center justify-center",
                    txn.type === "earning" || txn.type === "deposit" 
                      ? "bg-emerald-500/10" 
                      : "bg-red-500/10"
                  )}>
                    {txn.type === "earning" || txn.type === "deposit" ? (
                      <TrendingUp className="w-5 h-5 text-emerald-500" />
                    ) : (
                      <TrendingDown className="w-5 h-5 text-red-500" />
                    )}
                  </div>
                  <div>
                    <p className="font-bold text-sm">{txn.source}</p>
                    <p className="text-xs text-white/40">{txn.user} • {txn.time}</p>
                  </div>
                </div>
                <div className="text-right">
                  <p className={cn(
                    "text-lg font-bold",
                    txn.amount > 0 ? "text-emerald-500" : "text-red-500"
                  )}>
                    {txn.amount > 0 ? "+" : ""}${Math.abs(txn.amount).toFixed(2)}
                  </p>
                  <p className="text-xs text-white/40 capitalize">{txn.type}</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
}
