"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, MoreHorizontal, Star, Car, ShieldCheck, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, Loader2, Edit2, Plus } from "lucide-react";
import { useState, useEffect } from "react";
import { DriverService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function AdminDriversPage() {
  const router = useRouter();
  const [drivers, setDrivers] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState("all");
  const [searchTerm, setSearchTerm] = useState("");
  const [page, setPage] = useState(1);
  const [pagination, setPagination] = useState<any>(null);
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedDriver, setSelectedDriver] = useState<any>(null);

  useEffect(() => {
    fetchDrivers();
  }, [page, filter, searchTerm]);

  const fetchDrivers = async () => {
    setLoading(true);
    try {
      const params: any = { page };
      if (filter !== "all") {
        params.is_online = filter === "online" ? 1 : 0;
      }
      if (searchTerm) {
        params.search = searchTerm;
      }
      const response = await DriverService.getAll(params);
      const resData = response.data;
      setDrivers(resData.data || []);
      setPagination(resData.meta || resData);
    } catch (error) {
      console.error("Failed to fetch drivers:", error);
      toast.error("Failed to fetch drivers");
    } finally {
      setLoading(false);
    }
  };

  const handleSearch = (e: React.ChangeEvent<HTMLInputElement>) => {
    setSearchTerm(e.target.value);
    setPage(1);
  };

  const handleDeleteClick = (driver: any) => {
    setSelectedDriver(driver);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedDriver) return;
    try {
      await DriverService.delete(selectedDriver.id);
      toast.success("Driver deleted successfully");
      setDeleteModalOpen(false);
      setSelectedDriver(null);
      fetchDrivers();
    } catch (error) {
      console.error("Failed to delete driver:", error);
      toast.error("Failed to delete driver");
    }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Driver Management" 
        description="Manage driver profiles, approvals, and vehicle details."
        actionLabel="Add New Driver"
        actionLink="/admin/drivers/create"
      />

       <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
            {[
               { id: "all", label: "All Drivers" },
               { id: "online", label: "Online" },
               { id: "offline", label: "Offline" }
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => { setFilter(tab.id); setPage(1); }}
                className={`px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all ${filter === tab.id ? 'shadow-sm' : ''}`}
                style={{
                  fontSize: '0.85rem',
                  backgroundColor: filter === tab.id ? 'var(--text-color)' : 'transparent',
                  color: filter === tab.id ? 'var(--bg-color)' : 'var(--muted-text)',
                }}
              >
                {tab.label}
              </button>
            ))}
          </div>
          
          <div className="position-relative" style={{ width: '280px' }}>
            <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
            <input 
              type="text" 
              placeholder="Search drivers..." 
              value={searchTerm}
              onChange={handleSearch}
              className="form-control ps-5 py-2"
              style={{ 
                borderRadius: '12px',
                backgroundColor: 'var(--bg-color)',
                borderColor: 'var(--border-color)',
                color: 'var(--text-color)',
                fontSize: '0.9rem'
              }}
            />
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          {loading ? (
             <div className="d-flex flex-column align-items-center justify-content-center py-5 w-100">
                <Loader2 className="spin-animation text-primary mb-2" size={32} />
                <p className="text-muted-custom small">Loading drivers...</p>
             </div>
          ) : drivers.length === 0 ? (
            <div className="d-flex flex-column align-items-center justify-content-center py-5 opacity-50 w-100">
               <Car size={48} className="mb-2" />
               <p>No drivers found.</p>
            </div>
          ) : (
            <>
              <table className="table mb-0 align-middle">
                <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
                  <tr>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Driver</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Vehicle & Mode</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Rating</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Wallet</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Status</th>
                    <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {drivers.map((driver) => (
                    <tr key={driver.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                      <td className="px-4 py-3">
                        <div className="d-flex align-items-center gap-3">
                          <div className="position-relative">
                            {driver.user?.mediaurls?.images?.[0]?.default ? (
                              <img 
                                src={driver.user.mediaurls.images[0].default} 
                                alt={driver.user.name}
                                className="rounded-circle border"
                                style={{ width: '40px', height: '40px', objectFit: 'cover' }}
                              />
                            ) : (
                              <div className="rounded-circle d-flex align-items-center justify-content-center fw-bold" style={{ 
                                width: '40px', 
                                height: '40px', 
                                background: 'linear-gradient(135deg, #a855f7 0%, #7e22ce 100%)',
                                color: 'white'
                              }}>
                                {driver.user?.name?.substring(0, 1) || "?"}
                              </div>
                            )}
                            {(driver.is_verified === 1 || driver.user?.is_verified === 1) && (
                              <div className="position-absolute bottom-0 end-0 bg-white rounded-circle d-flex align-items-center justify-content-center border shadow-sm" style={{ width: '16px', height: '16px' }}>
                                <ShieldCheck size={12} className="text-primary" />
                              </div>
                            )}
                          </div>
                          <div>
                            <p className="fw-bold mb-0" style={{ color: 'var(--text-color)' }}>{driver.user?.name || "Unknown"}</p>
                            <p className="small mb-0 text-muted-custom">{driver.user?.mobile_number || driver.user?.mobile || "No mobile"}</p>
                          </div>
                        </div>
                      </td>
                      <td className="px-4 py-3">
                        <div className="d-flex flex-column gap-1">
                          <div className="d-flex align-items-center gap-2 text-muted-custom">
                             <Car size={14} className="text-primary-custom" />
                             <span className="small fw-medium">
                               {driver.vehicletypes?.[0]?.title || driver.meta?.vehicle_brand || "No Vehicle"}
                             </span>
                          </div>
                          <div className="d-flex gap-1 flex-wrap">
                            {driver.meta?.vehicle_number && (
                              <span className="extra-small text-muted-custom bg-light px-2 py-0.5 rounded border" style={{ fontSize: '0.7rem' }}>
                                {driver.meta.vehicle_number}
                              </span>
                            )}
                            {driver.meta?.riding_mode && (
                              <span className="extra-small text-primary-custom bg-primary bg-opacity-10 px-2 py-0.5 rounded border border-primary border-opacity-20 text-capitalize" style={{ fontSize: '0.7rem' }}>
                                {driver.meta.riding_mode}
                              </span>
                            )}
                          </div>
                        </div>
                      </td>
                      <td className="px-4 py-3">
                        <div className="d-flex align-items-center gap-1">
                          <Star size={14} className="text-warning fill-warning" />
                          <span className="small fw-bold" style={{ color: 'var(--text-color)' }}>
                            {Number(driver.ratings || driver.user?.ratings || 0).toFixed(1)}
                          </span>
                        </div>
                      </td>
                      <td className="px-4 py-3">
                        <div className="d-flex flex-column">
                          <span className="small fw-bold text-success">${Number(driver.user?.wallet?.balance || 0).toFixed(2)}</span>
                          <span className="extra-small text-muted-custom text-uppercase fw-bold" style={{ fontSize: '0.65rem' }}>Balance</span>
                        </div>
                      </td>
                      <td className="px-4 py-3">
                        <div className="d-flex flex-column gap-1">
                          <span 
                            className="badge rounded-pill fw-bold border"
                            style={{
                              backgroundColor: driver.is_online ? 'rgba(16, 185, 129, 0.1)' : 'rgba(107, 114, 128, 0.1)',
                              color: driver.is_online ? '#10b981' : '#6b7280',
                              borderColor: driver.is_online ? 'rgba(16, 185, 129, 0.2)' : 'rgba(107, 114, 128, 0.2)',
                              padding: '0.4em 0.8em',
                              width: 'fit-content'
                            }}
                          >
                            {driver.is_online && <span className="d-inline-block rounded-circle me-1" style={{ width: '6px', height: '6px', backgroundColor: 'currentColor' }}></span>}
                            {driver.is_online ? "Online" : "Offline"}
                          </span>
                          {driver.status && (
                            <span className="extra-small text-muted-custom text-capitalize ps-1" style={{ fontSize: '0.75rem' }}>
                              {driver.status}
                            </span>
                          )}
                        </div>
                      </td>
                      <td className="px-4 py-3 text-end">
                        <ActionMenu 
                          onView={() => router.push(`/admin/drivers/${driver.id}`)}
                          onEdit={() => router.push(`/admin/drivers/${driver.id}/edit`)}
                          onDelete={() => handleDeleteClick(driver)}
                        />
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>

              {/* Pagination UI */}
              {pagination && pagination.last_page > 1 && (
                <div className="p-4 border-top d-flex justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
                  <div className="small text-muted-custom">
                    Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> results
                  </div>
                  <div className="d-flex gap-2 flex-wrap">
                    {/* First Page Button */}
                    <button
                      onClick={() => setPage(1)}
                      disabled={pagination.current_page === 1}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                      title="First Page"
                    >
                      <ChevronsLeft size={16} />
                    </button>
                    
                    {/* Previous Button */}
                    <button
                      onClick={() => setPage(prev => Math.max(prev - 1, 1))}
                      disabled={pagination.current_page === 1}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                      title="Previous Page"
                    >
                      <ChevronLeft size={16} />
                    </button>
                    
                    {/* Page Numbers */}
                    <div className="d-flex align-items-center gap-1 flex-wrap">
                      {pagination.links
                        ?.filter((link: any) => {
                          const pageNum = parseInt(link.label);
                          return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                        })
                        .map((link: any, index: number) => {
                          const isCurrent = link.active;
                          const pageNumber = parseInt(link.label);
                          
                          return (
                            <button
                              key={index}
                              onClick={() => setPage(pageNumber)}
                              disabled={isCurrent}
                              className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border'}`}
                              style={{ minWidth: '32px', height: '32px', padding: '0 0.5rem' }}
                            >
                              {link.label}
                            </button>
                          );
                        })}
                    </div>
                    
                    {/* Next Button */}
                    <button
                      onClick={() => setPage(prev => Math.min(prev + 1, pagination.last_page))}
                      disabled={pagination.current_page === pagination.last_page}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                      title="Next Page"
                    >
                      <ChevronRight size={16} />
                    </button>
                    
                    {/* Last Page Button */}
                    <button
                      onClick={() => setPage(pagination.last_page)}
                      disabled={pagination.current_page === pagination.last_page}
                      className="btn btn-white border btn-sm d-flex align-items-center justify-content-center p-2 rounded-3 shadow-sm transition-all"
                      title="Last Page"
                    >
                      <ChevronsRight size={16} />
                    </button>
                  </div>
                </div>
              )}
            </>
          )}
        </div>
      </div>

      <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={selectedDriver?.user?.name || undefined}
        title="Delete Driver"
        description="Are you sure you want to delete this driver? This action cannot be undone."
      />
    </div>
  );
}
