"use client";

import PageHeader from "@/components/admin/PageHeader";
import { Save, X, User, Car, ShieldCheck, MapPin, Loader2, Phone, Mail, Lock, Settings, Info } from "lucide-react";
import Link from "next/link";
import { useState } from "react";
import { DriverService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function CreateDriverPage() {
  const router = useRouter();
  const [loading, setLoading] = useState(false);

  // Form State
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    mobile_number: "",
    password: "",
    riding_mode: "courier",
    gender: "male",
    ride_serv_pref: "package",
    vehicle_number: "",
    hot_food_bag: "false",
    is_verified: 0,
    current_latitude: "",
    current_longitude: "",
    status: "pending",
  });

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value, type } = e.target as HTMLInputElement;
    setFormData((prev: any) => ({
      ...prev,
      [name]: type === "checkbox" ? ((e.target as HTMLInputElement).checked ? 1 : 0) : value,
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const payload: any = {
        name: formData.name,
        email: formData.email,
        mobile_number: formData.mobile_number,
        password: formData.password,
        is_verified: formData.is_verified,
        status: formData.status,
        meta: JSON.stringify({
          riding_mode: formData.riding_mode,
          gender: formData.gender,
          ride_serv_pref: formData.ride_serv_pref,
          vehicle_number: formData.vehicle_number,
          hot_food_bag: formData.hot_food_bag,
        }),
      };

      if (formData.current_latitude) payload.current_latitude = formData.current_latitude;
      if (formData.current_longitude) payload.current_longitude = formData.current_longitude;

      await DriverService.create(payload);
      toast.success("Driver created successfully!");
      router.push("/admin/drivers");
    } catch (err: any) {
      console.error(err);
      toast.error(err.response?.data?.message || "Failed to create driver");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="py-2">
      <PageHeader title="Create New Driver" description="Onboard a new driver to the fleet." />

      <form onSubmit={handleSubmit}>
        <div className="row g-4">
          
          {/* LEFT COLUMN: Personal & Vehicle Details */}
          <div className="col-12 col-lg-8">
            
            {/* 1. Account Identity */}
            <div className="card-custom p-4 mb-4">
              <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                <User size={20} className="text-primary-custom" />
                Personal Information
              </h5>
              
              <div className="row g-3">
                <div className="col-12 col-md-6">
                  <label className="form-label small fw-bold text-muted-custom">Full Name</label>
                  <div className="input-group">
                    <span className="input-group-text bg-transparent border-end-0"><User size={16} /></span>
                    <input type="text" name="name" value={formData.name} onChange={handleChange} className="form-control border-start-0 ps-0" placeholder="John Doe" required />
                  </div>
                </div>
                <div className="col-12 col-md-6">
                  <label className="form-label small fw-bold text-muted-custom">Email Address</label>
                  <div className="input-group">
                    <span className="input-group-text bg-transparent border-end-0"><Mail size={16} /></span>
                    <input type="email" name="email" value={formData.email} onChange={handleChange} className="form-control border-start-0 ps-0" placeholder="john@example.com" required />
                  </div>
                </div>
                <div className="col-12 col-md-6">
                  <label className="form-label small fw-bold text-muted-custom">Mobile Number</label>
                  <div className="input-group">
                    <span className="input-group-text bg-transparent border-end-0"><Phone size={16} /></span>
                    <input type="text" name="mobile_number" value={formData.mobile_number} onChange={handleChange} className="form-control border-start-0 ps-0" placeholder="+20..." required />
                  </div>
                </div>
                <div className="col-12 col-md-6">
                  <label className="form-label small fw-bold text-muted-custom">Initial Password</label>
                  <div className="input-group">
                    <span className="input-group-text bg-transparent border-end-0"><Lock size={16} /></span>
                    <input type="password" name="password" value={formData.password} onChange={handleChange} className="form-control border-start-0 ps-0" placeholder="••••••••" required />
                  </div>
                </div>
                <div className="col-12 col-md-6">
                  <label className="form-label small fw-bold text-muted-custom">Gender</label>
                  <select name="gender" value={formData.gender} onChange={handleChange} className="form-select">
                    <option value="male">Male</option>
                    <option value="female">Female</option>
                  </select>
                </div>
              </div>
            </div>

            {/* 2. Vehicle Configuration */}
            <div className="card-custom p-4 mb-4">
              <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                <Car size={20} className="text-primary-custom" />
                Vehicle Specification
              </h5>
              
              <div className="row g-3">
                <div className="col-12 col-md-6">
                  <label className="form-label small fw-bold text-muted-custom">Riding Mode</label>
                  <select name="riding_mode" value={formData.riding_mode} onChange={handleChange} className="form-select">
                    <option value="courier">Courier (Pick-up/Delivery)</option>
                    <option value="taxi">Taxi (Rideshare)</option>
                    <option value="delivery">Heavy Delivery</option>
                  </select>
                </div>
                <div className="col-12 col-md-6">
                  <label className="form-label small fw-bold text-muted-custom">Service Preference</label>
                  <select name="ride_serv_pref" value={formData.ride_serv_pref} onChange={handleChange} className="form-select">
                    <option value="package">Packages Only</option>
                    <option value="ride">Rides Only</option>
                    <option value="both">Both Services</option>
                  </select>
                </div>
                <div className="col-12 col-md-6">
                  <label className="form-label small fw-bold text-muted-custom">Vehicle Number / Plate</label>
                  <input type="text" name="vehicle_number" value={formData.vehicle_number} onChange={handleChange} className="form-control" placeholder="ABC-123" />
                </div>
                <div className="col-12 col-md-6">
                  <label className="form-label small fw-bold text-muted-custom">Hot Food Bag Equipped?</label>
                  <select name="hot_food_bag" value={formData.hot_food_bag} onChange={handleChange} className="form-select">
                    <option value="true">Yes, Equipped</option>
                    <option value="false">No bag</option>
                  </select>
                </div>
              </div>
            </div>

          </div>

          {/* RIGHT COLUMN: Status & Geolocation */}
          <div className="col-12 col-lg-4">
            
            {/* 3. Account Status */}
            <div className="card-custom p-4 mb-4">
              <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                <Settings size={20} className="text-primary-custom" />
                Account Control
              </h5>
              
              <div className="mb-4">
                <label className="form-label small fw-bold text-muted-custom">Current Status</label>
                <select name="status" value={formData.status} onChange={handleChange} className="form-select">
                  <option value="pending">Pending Approval</option>
                  <option value="approved">Active / Approved</option>
                  <option value="rejected">Rejected</option>
                  <option value="suspended">Suspended</option>
                </select>
              </div>

              <div className="bg-light p-3 rounded-3 border mb-2" style={{ backgroundColor: 'var(--hover-bg)' }}>
                <div className="form-check form-switch d-flex align-items-center justify-content-between p-0">
                  <label className="form-check-label small fw-bold" htmlFor="isVerified">
                    Verified ID Status
                  </label>
                  <input 
                    className="form-check-input ms-0" 
                    type="checkbox" 
                    id="isVerified" 
                    name="is_verified" 
                    checked={formData.is_verified === 1} 
                    onChange={handleChange} 
                    style={{ width: '40px', height: '20px' }}
                  />
                </div>
                <p className="extra-small text-muted mb-0 mt-2">Display verification badge on customer apps.</p>
              </div>
            </div>

            {/* 4. Geolocation (Simulated) */}
            <div className="card-custom p-4 mb-4">
              <h5 className="fw-bold mb-4 d-flex align-items-center gap-2" style={{ color: 'var(--text-color)' }}>
                <MapPin size={20} className="text-primary-custom" />
                Initial Location
              </h5>
              
              <div className="row g-3">
                <div className="col-12">
                  <label className="form-label small fw-bold text-muted-custom">Latitude</label>
                  <input type="text" name="current_latitude" value={formData.current_latitude} onChange={handleChange} className="form-control" placeholder="30.0444" />
                </div>
                <div className="col-12">
                  <label className="form-label small fw-bold text-muted-custom">Longitude</label>
                  <input type="text" name="current_longitude" value={formData.current_longitude} onChange={handleChange} className="form-control" placeholder="31.2357" />
                </div>
              </div>
              
              <div className="mt-4 p-3 rounded-3 bg-info bg-opacity-10 border border-info border-opacity-20 d-flex gap-2">
                <Info size={16} className="text-info mt-1 flex-shrink-0" />
                <p className="extra-small text-info mb-0">Initial coordinates are optional. Drivers will update their location via the mobile app during work.</p>
              </div>
            </div>

            {/* Actions Footer */}
            <div className="d-flex flex-column gap-2">
              <button 
                type="submit" 
                disabled={loading} 
                className="btn btn-primary fw-bold py-3 shadow-sm d-flex align-items-center justify-content-center gap-2 border-0"
                style={{ background: 'linear-gradient(135deg, #f97316 0%, #ea580c 100%)' }}
              >
                {loading ? <Loader2 size={18} className="spin-animation" /> : <Save size={18} />}
                SAVE & PUBLISH DRIVER
              </button>
              <Link href="/admin/drivers" className="btn btn-light fw-bold py-2 border shadow-sm">
                Discard Changes
              </Link>
            </div>
            
          </div>
        </div>
      </form>
    </div>
  );
}
