"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import DeleteModal from "@/components/admin/DeleteModal";
import { Search, Layers, RefreshCw, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight } from "lucide-react";
import { useState, useEffect } from "react";
import { CategoryService } from "@/services/api";
import toast from "react-hot-toast";
import { useRouter } from "next/navigation";

export default function CategoriesPage() {
  const router = useRouter();
  const [categories, setCategories] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [filter, setFilter] = useState("all");
  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [selectedCategory, setSelectedCategory] = useState<any | null>(null);
  const [pagination, setPagination] = useState<any>(null);
  const [searchTerm, setSearchTerm] = useState("");

  useEffect(() => {
    fetchCategories(1);
  }, [filter]);

  const fetchCategories = async (page = 1) => {
    try {
      setLoading(true);
      const params: any = { page };
      if (filter !== "all") params.scope = filter;
      
      const response = await CategoryService.getAll(params);
      setCategories(response.data.data || []);
      setPagination(response.data.meta);
    } catch (err) {
      console.error("Failed to fetch categories:", err);
      toast.error("Failed to load categories");
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteClick = (category: any) => {
    setSelectedCategory(category);
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!selectedCategory) return;
    
    try {
        await CategoryService.delete(selectedCategory.id);
        toast.success("Category deleted successfully");
        fetchCategories(); // Refresh list
    } catch (err) {
        console.error("Failed to delete category:", err);
        toast.error("Failed to delete category");
    } finally {
        setDeleteModalOpen(false);
        setSelectedCategory(null);
    }
  };

  const filteredCategories = categories.filter(cat => 
    (cat.title || cat.name || '').toLowerCase().includes(searchTerm.toLowerCase()) ||
    (cat.slug || '').toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <div className="py-2">
      <PageHeader 
        title="Category Management" 
        description="Manage product categories, hierarchy, and scopes."
        actionLabel="Add New Category"
        actionLink="/admin/categories/create"
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)' }}>
            {["all", "home", "ecommerce"].map((tab) => (
              <button
                key={tab}
                onClick={() => setFilter(tab)}
                className={`px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all text-nowrap ${filter === tab ? 'shadow-sm' : ''}`}
                style={{
                  fontSize: '0.85rem',
                  backgroundColor: filter === tab ? 'var(--text-color)' : 'transparent',
                  color: filter === tab ? 'var(--bg-color)' : 'var(--muted-text)',
                }}
              >
                {tab.split('_').join(' ')}
              </button>
            ))}
          </div>
          
          <div className="d-flex gap-2 w-100 w-md-auto">
             <div className="position-relative flex-grow-1" style={{ width: '280px' }}>
                <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
                <input 
                  type="text" 
                  placeholder="Search categories..." 
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="form-control ps-5 py-2"
                  style={{ 
                      borderRadius: '12px',
                      backgroundColor: 'var(--bg-color)',
                      borderColor: 'var(--border-color)',
                      color: 'var(--text-color)',
                      fontSize: '0.9rem'
                  }}
                />
            </div>
            <button 
                onClick={() => fetchCategories(pagination?.current_page || 1)} 
                className="btn btn-light border d-flex align-items-center justify-content-center"
                style={{ width: '40px', height: '40px', borderColor: 'var(--border-color)' }}
                title="Refresh List"
            >
                <RefreshCw size={18} className={loading ? "animate-spin" : ""} />
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          <table className="table mb-0 align-middle">
            <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
              <tr>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Category</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Type & Scope</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Status</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                  <tr>
                      <td colSpan={4} className="text-center py-5">
                          <div className="spinner-border text-primary" role="status">
                            <span className="visually-hidden">Loading...</span>
                          </div>
                      </td>
                  </tr>
              ) : filteredCategories.length === 0 ? (
                  <tr>
                      <td colSpan={4} className="text-center py-5">
                          <Layers size={48} className="text-muted opacity-25 mb-2" />
                          <p className="text-muted mb-0">No categories found.</p>
                      </td>
                  </tr>
              ) : (
                filteredCategories.map((category) => {
                    const imageUrl = category.mediaurls?.images?.[0]?.default || category.mediaurls?.banners?.[0]?.default;
                    
                    return (
                        <tr key={category.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                        <td className="px-4 py-3">
                            <div className="d-flex align-items-center gap-3">
                            <div className="rounded-3 d-flex align-items-center justify-content-center overflow-hidden position-relative" style={{ 
                                width: '50px', 
                                height: '50px', 
                                backgroundColor: 'var(--hover-bg)',
                                color: 'var(--muted-text)',
                                border: '1px solid var(--border-color)'
                            }}>
                                {imageUrl ? (
                                    <img 
                                      src={imageUrl} 
                                      alt={category.title} 
                                      className="w-100 h-100 object-fit-cover position-absolute top-0 start-0"
                                      onError={(e) => {
                                        (e.target as HTMLImageElement).style.display = 'none';
                                        const parent = (e.target as HTMLImageElement).parentElement;
                                        if (parent) {
                                          const fallback = parent.querySelector('.fallback-icon');
                                          if (fallback) fallback.classList.remove('d-none');
                                        }
                                      }}
                                    />
                                ) : null}
                                <Layers size={20} className={`text-muted ${imageUrl ? 'd-none' : ''} fallback-icon`} />
                            </div>
                            <div>
                                <p className="fw-bold mb-0" style={{ color: 'var(--text-color)' }}>
                                  {category.title || category.name || 'Unnamed'}
                                </p>
                                {category.slug && (
                                  <p className="small mb-0 text-muted-custom">/{category.slug}</p>
                                )}
                                {category.parent_id && (
                                  <span className="badge bg-secondary bg-opacity-10 text-secondary border border-secondary-subtle" style={{ fontSize: '0.7rem' }}>
                                    Sub-category
                                  </span>
                                )}
                            </div>
                            </div>
                        </td>
                        <td className="px-4 py-3">
                          <div className="d-flex flex-column gap-1">
                            {category.meta?.vendor_type && (
                              <span className="badge bg-info bg-opacity-10 text-info border border-info-subtle" style={{ width: 'fit-content', fontSize: '0.75rem' }}>
                                {category.meta.vendor_type}
                              </span>
                            )}
                            {category.meta?.scope && (
                              <span className="badge bg-primary bg-opacity-10 text-primary border border-primary-subtle" style={{ width: 'fit-content', fontSize: '0.75rem' }}>
                                {category.meta.scope}
                              </span>
                            )}
                            {category.meta?.has_takeaway && (
                              <span className="badge bg-success bg-opacity-10 text-success border border-success-subtle" style={{ width: 'fit-content', fontSize: '0.75rem' }}>
                                Takeaway
                              </span>
                            )}
                          </div>
                        </td>
                        <td className="px-4 py-3">
                          <span 
                            className="badge rounded-pill fw-bold border"
                            style={{
                              backgroundColor: (category.meta?.is_enabled || category.active || category.status === "Active") ? 'rgba(16, 185, 129, 0.1)' : 'rgba(107, 114, 128, 0.1)',
                              color: (category.meta?.is_enabled || category.active || category.status === "Active") ? '#10b981' : '#6b7280',
                              borderColor: (category.meta?.is_enabled || category.active || category.status === "Active") ? 'rgba(16, 185, 129, 0.2)' : 'rgba(107, 114, 128, 0.2)',
                              padding: '0.5em 0.8em'
                            }}
                          >
                            {(category.meta?.is_enabled || category.active || category.status === "Active") ? "Enabled" : "Disabled"}
                          </span>
                        </td>
                        <td className="px-4 py-3 text-end">
                            <ActionMenu 
                                onView={() => router.push(`/admin/categories/${category.id}`)}
                                onEdit={() => router.push(`/admin/categories/${category.id}/edit`)}
                                onDelete={() => handleDeleteClick(category)}
                            />
                        </td>
                        </tr>
                    );
                })
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Pagination */}
      {pagination && (
        <div className="d-flex justify-content-between align-items-center mt-4">
          <div className="small text-muted-custom">
            Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> results
          </div>
          <div className="d-flex gap-2 flex-wrap">
            <button
              onClick={() => fetchCategories(1)}
              disabled={pagination.current_page === 1}
              className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
              style={{ padding: '0.5rem 1rem' }}
              title="Go to first page"
            >
              <ChevronsLeft size={16} /> First
            </button>
            <button
              onClick={() => fetchCategories(pagination.current_page - 1)}
              disabled={pagination.current_page === 1}
              className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
              style={{ padding: '0.5rem 1rem' }}
            >
              <ChevronLeft size={16} /> Previous
            </button>
            <div className="d-flex align-items-center gap-1 flex-wrap">
              {pagination.links
                ?.filter((link: any) => {
                  const pageNum = parseInt(link.label);
                  return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                })
                .map((link: any, index: number) => {
                  const isCurrent = link.active;
                  const pageNumber = parseInt(link.label);
                  return (
                    <button
                      key={index}
                      onClick={() => fetchCategories(pageNumber)}
                      disabled={isCurrent}
                      className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border'}`}
                      style={{ minWidth: '32px', height: '32px', padding: '0 0.5rem' }}
                    >
                      {link.label}
                    </button>
                  );
                })}
            </div>
            <button
              onClick={() => fetchCategories(pagination.current_page + 1)}
              disabled={pagination.current_page === pagination.last_page}
              className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
              style={{ padding: '0.5rem 1rem' }}
            >
              Next <ChevronRight size={16} />
            </button>
            <button
              onClick={() => fetchCategories(pagination.last_page)}
              disabled={pagination.current_page === pagination.last_page}
              className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
              style={{ padding: '0.5rem 1rem' }}
              title="Go to last page"
            >
              Last <ChevronsRight size={16} />
            </button>
          </div>
        </div>
      )}
      
       <DeleteModal 
        isOpen={deleteModalOpen}
        onClose={() => setDeleteModalOpen(false)}
        onConfirm={confirmDelete}
        itemName={selectedCategory?.title || selectedCategory?.name}
        title="Delete Category"
        description="Are you sure you want to delete this category? This action cannot be undone."
      />
    </div>
  );
}
