"use client";

import PageHeader from "@/components/admin/PageHeader";
import ActionMenu from "@/components/admin/ActionMenu";
import { Search, RefreshCw, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, Calendar, User, Truck, Clock, CreditCard, DollarSign } from "lucide-react";
import { useState, useEffect } from "react";
import { AppointmentService } from "@/services/api";
import { useRouter } from "next/navigation";
import toast from "react-hot-toast";

export default function AdminAppointmentsPage() {
  const router = useRouter();
  const [appointments, setAppointments] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [filter, setFilter] = useState("all");
  const [page, setPage] = useState(1);
  const [pagination, setPagination] = useState<any>(null);

  useEffect(() => {
    fetchAppointments(page);
  }, [page]); // Removed filter from dependency to avoid refetching

  const fetchAppointments = async (newPage = 1) => {
    setLoading(true);
    try {
      const params: any = { page: newPage };
      if (searchTerm) {
        params.search = searchTerm;
      }
      // Removed backend status filtering
      // if (filter !== "all") {
      //   params.status = filter;
      // }
      const response = await AppointmentService.getAll(params);
      const resData = response.data;
      setAppointments(resData.data || []);
      setPagination(resData.meta || resData);
      setPage(newPage);
    } catch (error) {
      console.error("Failed to fetch appointments:", error);
      toast.error("Failed to load appointments");
    } finally {
      setLoading(false);
    }
  };

  // Client-side filtering logic
  const filteredAppointments = appointments.filter(item => {
    if (filter === "all") return true;
    return item.status?.toLowerCase() === filter.toLowerCase();
  });

  const getStatusBadge = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'accepted': case 'ongoing': case 'onway':
        return <span className="badge rounded-pill fw-bold border bg-success-subtle text-success border-success-subtle px-3 py-2">Accepted</span>;
      case 'pending':
        return <span className="badge rounded-pill fw-bold border bg-warning-subtle text-warning border-warning-subtle px-3 py-2">Pending</span>;
      case 'cancelled': case 'rejected':
        return <span className="badge rounded-pill fw-bold border bg-danger-subtle text-danger border-danger-subtle px-3 py-2">Cancelled</span>;
      case 'complete': case 'completed':
        return <span className="badge rounded-pill fw-bold border bg-primary-subtle text-primary border-primary-subtle px-3 py-2">Completed</span>;
      default:
        return <span className="badge rounded-pill fw-bold border bg-secondary-subtle text-secondary border-secondary-subtle px-3 py-2">{status}</span>;
    }
  };

  return (
    <div className="py-2">
      <PageHeader 
        title="Appointments" 
        description="Manage service appointments and track their status."
      />

      <div className="card-custom overflow-hidden">
        {/* Toolbar */}
        <div className="p-4 border-bottom d-flex flex-column flex-md-row gap-3 justify-content-between align-items-center" style={{ borderColor: 'var(--border-color)' }}>
          <div className="d-flex p-1 rounded-3 overflow-auto" style={{ backgroundColor: 'var(--bg-color)', border: '1px solid var(--border-color)', maxWidth: '100%' }}>
             {["all", "pending", "accepted", "ongoing", "complete", "cancelled"].map((tab) => (
              <button
                key={tab}
                onClick={() => { setFilter(tab); }}
                className={`px-3 py-1.5 rounded-2 text-capitalize fw-bold border-0 transition-all text-nowrap ${filter === tab ? 'shadow-sm' : ''}`}
                style={{
                  fontSize: '0.85rem',
                  backgroundColor: filter === tab ? 'var(--text-color)' : 'transparent',
                  color: filter === tab ? 'var(--bg-color)' : 'var(--muted-text)',
                }}
              >
                {tab}
              </button>
            ))}
          </div>

          <div className="d-flex gap-2 w-100 w-md-auto">
             <div className="position-relative flex-grow-1" style={{ width: '280px' }}>
                <Search className="position-absolute top-50 translate-middle-y ms-3" size={16} style={{ color: 'var(--muted-text)' }} />
                <input 
                  type="text" 
                  placeholder="Search appointments..." 
                  value={searchTerm}
                  onChange={(e) => { setSearchTerm(e.target.value); }}
                  onKeyDown={(e) => { if (e.key === 'Enter') fetchAppointments(1); }}
                  className="form-control ps-5 py-2"
                  style={{ 
                    borderRadius: '12px',
                    backgroundColor: 'var(--bg-color)',
                    borderColor: 'var(--border-color)',
                    color: 'var(--text-color)',
                    fontSize: '0.9rem'
                  }}
                />
            </div>
            <button 
                onClick={() => fetchAppointments(pagination?.current_page || 1)} 
                className="btn btn-light border d-flex align-items-center justify-content-center"
                style={{ width: '40px', height: '40px', borderColor: 'var(--border-color)' }}
                title="Refresh List"
            >
                <RefreshCw size={18} className={loading ? "animate-spin" : ""} />
            </button>
          </div>
        </div>

        {/* Table */}
        <div className="table-responsive">
          <table className="table mb-0 align-middle">
            <thead style={{ backgroundColor: 'var(--hover-bg)' }}>
              <tr>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>ID</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Service Info</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Parties</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Schedule</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Status</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom" style={{ borderColor: 'var(--border-color)' }}>Payment</th>
                <th className="px-4 py-3 small text-uppercase text-muted-custom border-bottom text-end" style={{ borderColor: 'var(--border-color)' }}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {loading ? (
                 <tr>
                    <td colSpan={7} className="text-center py-5">
                       <div className="spinner-border text-primary" role="status">
                           <span className="visually-hidden">Loading...</span>
                       </div>
                    </td>
                 </tr>
              ) : filteredAppointments.length === 0 ? (
                <tr>
                   <td colSpan={7} className="text-center py-5 text-muted">
                      {filter === "all" ? "No appointments found." : `No ${filter} appointments found on this page.`}
                   </td>
                </tr>
              ) : (
                filteredAppointments.map((item) => (
                  <tr key={item.id} style={{ borderBottom: '1px solid var(--border-color)' }}>
                    <td className="px-4 py-3">
                      <span className="small text-muted-custom fw-bold">#{item.id}</span>
                    </td>
                    <td className="px-4 py-3">
                       <div className="d-flex flex-column" style={{ maxWidth: '220px' }}>
                          <span className="text-dark fw-bold text-truncate" title={item.meta?.categories?.[0]?.title || ""}>
                            {item.meta?.categories?.[0]?.title || "Service Request"}
                          </span>
                          <span className="text-muted-custom small text-truncate d-block mt-1">
                             {item.address || "No address provided"}
                          </span>
                       </div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="d-flex flex-column gap-1">
                        <div className="d-flex align-items-center gap-2" title="Customer">
                           <User size={14} className="text-muted-custom" />
                           <span className="text-dark small fw-medium">{item.user?.name || "Guest User"}</span>
                        </div>
                        <div className="d-flex align-items-center gap-2" title="Provider">
                           <Truck size={14} className="text-muted-custom" />
                           <span className="text-dark small">{item.provider?.name || "Unassigned"}</span>
                        </div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                       <div className="d-flex flex-column small">
                         <div className="d-flex align-items-center gap-2 text-dark fw-medium">
                           <Calendar size={14} className="text-primary-custom" /> 
                           {item.date}
                         </div>
                         <div className="d-flex align-items-center gap-2 text-muted-custom mt-1">
                           <Clock size={14} /> 
                           {item.time_from ? `${item.time_from.slice(0,5)} - ${item.time_to?.slice(0,5)}` : item.time || "--:--"}
                         </div>
                       </div>
                    </td>
                    <td className="px-4 py-3">
                       {getStatusBadge(item.status)}
                    </td>
                    <td className="px-4 py-3">
                       <div className="d-flex flex-column">
                          <span className="fw-bold text-dark d-flex align-items-center gap-1">
                             <DollarSign size={14} className="text-success" />
                             {Number(item.amount || item.total || 0).toFixed(2)}
                          </span>
                          <div className="d-flex align-items-center gap-1 mt-1 small">
                             <CreditCard size={12} className="text-muted-custom" />
                             <span className="text-muted-custom text-capitalize">
                                {item.payment?.payment_method?.slug || item.payment?.status || "Unpaid"}
                             </span>
                          </div>
                       </div>
                    </td>
                    <td className="px-4 py-3 text-end">
                      <ActionMenu 
                          onView={() => router.push(`/admin/appointments/${item.id}`)}
                          onDelete={() => {}} // Placeholder
                      />
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* Pagination */}
      {pagination && (
         <div className="d-flex justify-content-between align-items-center mt-4">
           <div className="small text-muted-custom">
             Showing <span className="fw-bold text-dark">{pagination.from || 0}</span> to <span className="fw-bold text-dark">{pagination.to || 0}</span> of <span className="fw-bold text-dark">{pagination.total || 0}</span> results
           </div>
           <div className="d-flex gap-2 flex-wrap">
             <button
               onClick={() => fetchAppointments(1)}
               disabled={pagination.current_page === 1}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               <ChevronsLeft size={16} /> First
             </button>
             <button
               onClick={() => fetchAppointments(pagination.current_page - 1)}
               disabled={pagination.current_page === 1}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               <ChevronLeft size={16} /> Previous
             </button>
             
             <div className="d-flex align-items-center gap-1 flex-wrap">
               {pagination.links
                 ?.filter((link: any) => {
                   const pageNum = parseInt(link.label);
                   return !isNaN(pageNum) && link.label.trim() === pageNum.toString();
                 })
                 .map((link: any, index: number) => {
                   const isCurrent = link.active;
                   const pageNumber = parseInt(link.label);
                   
                   return (
                     <button
                       key={index}
                       onClick={() => fetchAppointments(pageNumber)}
                       disabled={isCurrent}
                       className={`btn btn-sm ${isCurrent ? 'btn-primary' : 'btn-light border'}`}
                       style={{ minWidth: '32px', height: '32px', padding: '0 0.5rem' }}
                     >
                       {link.label}
                     </button>
                   );
                 })}
             </div>

             <button
               onClick={() => fetchAppointments(pagination.current_page + 1)}
               disabled={pagination.current_page === pagination.last_page}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               Next <ChevronRight size={16} />
             </button>
             <button
               onClick={() => fetchAppointments(pagination.last_page)}
               disabled={pagination.current_page === pagination.last_page}
               className="btn btn-white border d-flex align-items-center gap-2 shadow-sm"
               style={{ padding: '0.5rem 1rem' }}
             >
               Last <ChevronsRight size={16} />
             </button>
           </div>
         </div>
      )}
    </div>
  );
}
